package de.narimo.georepo.server.api;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.species.Species;
import de.narimo.georepo.server.api.species.SpeciesService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{workspace}/layers/{layername}/species")
@Tag(name = "Species")
public class SpeciesController {

    SpeciesService speciesService = new SpeciesService();

    @GET
    @Path("/")
    @Produces({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Retrieve all available species")
    public Response getSpecies(
            @Context SecurityContext sec,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName) {

        try {
            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);

            List<Species> species = speciesService.getSpecies(layerId);
            return Response.ok().entity(species).build();

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Add a new species")
    public Response addSpecies(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @PathParam("workspace") String workspace,
            @PathParam("layername") String dataLayerName,
            Species species) throws IOException, SQLException {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();
            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            speciesService.addSpecies(layerId, user, species);

        } catch (SQLException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        return Response.status(201).build();
    }
}
