package de.narimo.georepo.server.api;

import java.sql.SQLException;
import java.util.List;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.tags.TagService;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.tools.AdminTools;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces/{diffWorkspace}/layers/{layername}/features/{gfid}/tags")
@Tag(name = "Tags")
public class TagsDiffController {

    TagService tagService = new TagService();

    @POST
    @Path("/")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Suggest adding tags")
    public Response suggestAddFeatureTags(
            @Context SecurityContext sec,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            List<String> tags) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            int layerId = DatasetRepository.getLayerId(dataWorkspace, dataLayerName);
            // tags get directly written to the _tag table, which means,
            // they don't need review
            tagService.addTags(layerId, user.getId(), gfid, tags);
            return Response.ok().build();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @DELETE
    @Path("/")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Operation(summary = "Suggest removing tags")
    public Response suggestRemoveFeatureTags(
            @Context SecurityContext sec,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            List<String> tags) {

        try {

            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String dataWorkspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (dataWorkspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            int layerId = DatasetRepository.getLayerId(dataWorkspace, dataLayerName);
            // tags get directly written to the _tag table, which means,
            // they don't need review
            tagService.removeTags(layerId, user.getId(), gfid, tags);
            return Response.ok().build();
        } catch (SQLException | InternalError e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/{tagId}/accept")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Accept a suggested tag for a feature")
    public Response acceptTagChangeset(
            @Context SecurityContext sec,
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("tagId") int tagId) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            tagService.acceptTag(layerId, gfid, tagId);
            return Response.ok().build();

        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @PUT
    @Path("/{tagId}/decline")
    @Operation(summary = "Decline a suggested tag for a feature")
    public Response declineTagChangeset(
            @PathParam("diffWorkspace") String diffWorkspace,
            @PathParam("layername") String dataLayerName,
            @PathParam("gfid") int gfid,
            @PathParam("tagId") int tagId,
            @Context SecurityContext sec) {

        try {
            User user = UserRepository.getUser(sec.getUserPrincipal().getName()).get();

            String workspace = DifftableRepository.getDataWorkspace(diffWorkspace, dataLayerName);
            if (workspace == null) {
                throw new InternalError("No data workspace defined.");
            }

            AdminTools.checkAdminPermission(user.getId(), workspace);

            int layerId = DatasetRepository.getLayerId(workspace, dataLayerName);
            tagService.declineTag(layerId, gfid, tagId);
            return Response.ok().build();

        } catch (NumberFormatException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).entity("Invalid feature id.").build();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (NotFoundException e) {
            e.printStackTrace();
            return Response.status(Status.NOT_FOUND).build();
        } catch (ForbiddenException e) {
            e.printStackTrace();
            return Response.status(Status.FORBIDDEN).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}