package de.narimo.georepo.server.api.comments;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.ws.rs.NotFoundException;

import de.narimo.commons.dto.User;
import de.narimo.georepo.server.repository.CommentOrWeblinkRepository;
import de.narimo.georepo.server.repository.DatasetRepository;
import de.narimo.georepo.server.tools.TableTools;

public class CommentOrLinkService {

    public List<Weblink> getComments(User user, String workspace, int layerId, int gfid)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentOrWeblinkRepository.checkCommentsTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Comments table does not yet exist for layer " + layerId + ". No comments present.");
            return Arrays.asList();
        }

        return CommentOrWeblinkRepository.getFeatureCommentsOrWeblinks(user, workspace, layerId, gfid,
                CommentOrLinkType.COMMENT);
    }

    public List<Weblink> getUnreviewedComments(User user, String workspace, int layerId)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentOrWeblinkRepository.checkCommentsDiffTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Comments table does not yet exist for layer " + layerId + ". No comments present.");
            return Arrays.asList();
        }

        return CommentOrWeblinkRepository.getUnreviewedCommentsOrWeblinks(user, workspace, layerId,
                CommentOrLinkType.COMMENT);
    }

    public List<Weblink> getWeblinks(User user, String workspace, int layerId, int gfid)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentOrWeblinkRepository.checkCommentsTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println("Weblinks table does not yet exist for layer " + layerId + ". No weblinks present.");
            return Arrays.asList();
        }
        return CommentOrWeblinkRepository.getFeatureCommentsOrWeblinks(user, workspace, layerId, gfid,
                CommentOrLinkType.WEBLINK);
    }

    public List<Weblink> getUnreviewedWeblinks(User user, String workspace, int layerId)
            throws SQLException {

        // if no comments table exists, return empty list
        boolean commentsTableExists = CommentOrWeblinkRepository.checkCommentsDiffTableExists(layerId);
        if (!commentsTableExists) {
            System.out.println(
                    "Weblinks table does not yet exist for layer " + layerId + ". No suggested weblinks present.");
            return Arrays.asList();
        }
        return CommentOrWeblinkRepository.getUnreviewedCommentsOrWeblinks(user, workspace, layerId,
                CommentOrLinkType.WEBLINK);
    }

    public int addCommentOrWeblink(ServletContext ctx, User user, String workspace, int layerId, int gfid,
            CommentOrWeblink commentOrWeblink, CommentOrLinkType commentOrLinkType)
            throws SQLException {

        if (!DatasetRepository.doesFeatureExist(layerId, gfid)) {
            throw new NotFoundException();
        }

        boolean commentTableExists = CommentOrWeblinkRepository.checkCommentsTableExists(layerId);
        if (!commentTableExists) {
            System.out.println("Comment table does not yet exist for layer " + layerId + ". Creating it...");
            CommentOrWeblinkRepository.createCommentsTable(layerId);
        }

        boolean commentDiffTableExists = CommentOrWeblinkRepository.checkCommentsDiffTableExists(layerId);
        if (!commentDiffTableExists) {
            System.out.println("Comment diff table does not yet exist for layer " + layerId + ". Creating it...");
            CommentOrWeblinkRepository.createCommentDiffTable(layerId);
        }

        // TODO: Do we need to do more checks on the comments validity?
        if (!isCommentOrLinkValid(
                commentOrLinkType == CommentOrLinkType.COMMENT ? ((Comment) commentOrWeblink).getComment()
                        : ((Weblink) commentOrWeblink).getLink(),
                commentOrLinkType)) {
            throw new IllegalArgumentException("Comment contains invalid characters.");
        }

        String text = (commentOrLinkType == CommentOrLinkType.COMMENT)
                ? ((Comment) commentOrWeblink).getComment()
                : ((Weblink) commentOrWeblink).getLink();

        String description = (commentOrLinkType == CommentOrLinkType.WEBLINK)
                ? ((Weblink) commentOrWeblink).getDescription()
                : null;

        return CommentOrWeblinkRepository.insertUnreviewedCommentOrWeblink(layerId, user.getId(), gfid, text,
                commentOrLinkType,
                description);

//        inform the admins that a new comment has been uploaded to the workspace

//        boolean notifyAdmin = true;
//        if (notifyAdmin) {
//            Notifier.notifyWorkspaceAdminsAboutPOIChanges(ctx, dataWorkspace, dataLayerName);
//        }

    }

    public void acceptCommentOrWeblink(int layerId, int gfid, int commentId) {
        String commentsTable = TableTools.getCommentTableName(layerId);
        String diffCommentsTable = TableTools.getCommentDiffTableName(layerId);
        CommentOrWeblinkRepository.acceptCommentOrWeblink(commentsTable, diffCommentsTable, gfid, commentId);
    }

    public void declineCommentOrWeblink(int layerId, int gfid, int commentId) {
        String diffCommentsTable = TableTools.getCommentDiffTableName(layerId);
        CommentOrWeblinkRepository.declineCommentOrWeblink(diffCommentsTable, gfid, commentId);
    }

    private boolean isCommentOrLinkValid(String commentOrLink, CommentOrLinkType commentOrLinkType) {
        if (commentOrLinkType == CommentOrLinkType.COMMENT) {
            if (!(commentOrLink.contains("<") || commentOrLink.contains("script")
                    || commentOrLink.contains("iframe"))) {
                return true;
            }
        } else if (commentOrLinkType == CommentOrLinkType.WEBLINK) {
            if (!(commentOrLink.contains("<") || commentOrLink.contains("script")
                    || commentOrLink.contains("iframe"))) {
                return true;
            }
        }
        return false;
    }

    public void deleteCommentOrWeblink(int layerId, int gfid, int commentOrWeblinkId) {
        String commentsTable = TableTools.getCommentTableName(layerId);
        CommentOrWeblinkRepository.deleteCommentOrWeblink(commentsTable, gfid, commentOrWeblinkId);
    }
}
