package de.narimo.georepo.server.api.observations;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.repository.ParentDatasetRepository;
import de.narimo.georepo.server.tools.TableTools;

public class ObservationsRepository {

    public static boolean checkObservationsTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", obsTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Group all observations for a gfid by distinct speciesId and academic name in
     * the given time range and sum their occurrences.
     * 
     * @param layerId
     * @param gfid
     * @param fromDate
     * @param toDate
     * @return
     * @throws SQLException
     */
    public static List<Observation> getObservationsByTimeRange(int layerId, int gfid, String fromDate, String toDate)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);
        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT speciesid, nameacademic as speciesname, sum(amount)  FROM " + obsTable + " as o, "
                    + specTable + " as s "
                    + " WHERE o.speciesId = s.id and gfid = " + gfid
                    + " AND observedAt >= ?::timestamp AND observedAt <= ?::timestamp GROUP BY speciesId, nameacademic;";

            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setString(1, fromDate);
            ps.setString(2, toDate);
            ResultSet rs = ps.executeQuery();

            List<Observation> observations = new ArrayList<>();
            while (rs.next()) {
                Observation o = new Observation();
                o.setAmount(rs.getInt("sum"));
                o.setGfid(gfid);
                o.setSpeciesName(rs.getString("speciesname"));
                o.setSpeciesId(rs.getInt("speciesId"));
                observations.add(o);
            }
            return observations;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Group all observations for a gfid by distinct species id, academic name and
     * available years and sum their occurrences.
     * 
     * @param layerId
     * @param gfid
     * @return
     * @throws SQLException
     */
    public static List<Observation> getObservationsByYears(int layerId, int gfid)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);
        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT speciesid, nameacademic as speciesname, sum(amount), to_char(observedat, 'YYYY') as year FROM "
                    + obsTable
                    + " as o, " + specTable + " as s "
                    + " WHERE o.speciesId = s.id and gfid = " + gfid
                    + " GROUP BY speciesId, nameacademic, to_char(observedat, 'YYYY')";

            System.out.println(sql);

            ResultSet rs = jdbcData.executeQuery(sql);

            List<Observation> observations = new ArrayList<>();
            while (rs.next()) {
                Observation o = new Observation();
                o.setAmount(rs.getInt("sum"));
                o.setGfid(gfid);
                o.setSpeciesId(rs.getInt("speciesid"));
                o.setSpeciesName(rs.getString("speciesname"));
                o.setObservedAt(rs.getString("year"));
                observations.add(o);
            }
            return observations;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static List<Observation> getObservationsBySearchTerm(int layerId, String searchTerm)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);
        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT speciesid, nameacademic, amount, observedat, gfid FROM "
                    + obsTable + " as o, " + specTable + " as s "
                    + " WHERE o.speciesId = s.id AND s.nameacademic ilike ?";

            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setString(1, "%" + searchTerm + "%");

            ResultSet rs = ps.executeQuery();

            List<Observation> observations = new ArrayList<>();
            while (rs.next()) {
                Observation o = new Observation();
                o.setAmount(rs.getInt("amount"));
                o.setGfid(rs.getInt("gfid"));
                o.setSpeciesId(rs.getInt("speciesid"));
                o.setSpeciesName(rs.getString("nameacademic"));
                o.setObservedAt(rs.getString("observedat"));
                observations.add(o);
            }
            return observations;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void insertObservation(int layerId, Integer userId, Observation observation)
            throws SQLException {

        if (userId == null) {
            throw new RuntimeException();
        }

        if (observation.getSpeciesId() == null)
            throw new IllegalArgumentException("Parameter speciesId must not be null");

        if (observation.getGfid() == null) {
            throw new IllegalArgumentException("Parameter gfid must not be null");
        }

        if (observation.getAmount() == null) {
            throw new IllegalArgumentException("Parameter amount must not be null");
        }

        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            StringBuilder sb = new StringBuilder();
            sb.append("INSERT INTO " + obsTable + " (speciesId, gfid, amount, insertedby");

            if (observation.getObservedAt() != null) {
                sb.append(", observedAt");
            }
            sb.append(") VALUES (?, ?, ?, ?");

            if (observation.getObservedAt() != null) {
                sb.append(",?::timestamp");
            }
            sb.append(");");
            System.out.println(sb.toString());

            PreparedStatement ps = jdbcData.prepareStatement(sb.toString());
            ps.setInt(1, observation.getSpeciesId());
            ps.setInt(2, observation.getGfid());
            ps.setInt(3, observation.getAmount());
            ps.setInt(4, userId);
            if (observation.getObservedAt() != null) {
                ps.setString(5, observation.getObservedAt());
            }
            ps.executeUpdate();

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void createObservationsTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String obsTable = TableTools.getObservationsTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", obsTable)) {
                return;
            }
            System.out.println("Observations table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + obsTable + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "gfid integer NOT NULL,"
                    + "speciesId integer NOT NULL,"
                    + "amount integer NOT NULL,"
                    + "insertedby integer NOT NULL,"
                    + "observedAt timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
