package de.narimo.georepo.server.api.species;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.repository.ParentDatasetRepository;
import de.narimo.georepo.server.tools.TableTools;

public class SpeciesRepository {

    public static boolean checkSpeciesTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", specTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Should we return all available species?
     * 
     * @param layerId
     * @return
     * @throws SQLException
     */
    public static List<Species> getSpecies(int layerId)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT id, nameacademic, nameen, namees, iucnredlist, source FROM " + specTable + ";";
            System.out.println(sql);
            ResultSet rs = jdbcData.executeQuery(sql);

            List<Species> species = new ArrayList<>();
            while (rs.next()) {
                Species s = new Species();
                s.setId(rs.getInt("id"));
                s.setNameacademic(rs.getString("nameacademic"));
                s.setNameen(rs.getString("nameen"));
                s.setNamees(rs.getString("namees"));
                s.setSource(rs.getString("source"));
                if (rs.getBoolean("iucnredlist")) {
                    s.setIucnredlist(true);
                }
                species.add(s);
            }

            return species;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void insertSpecies(int layerId, Integer userId, Species species)
            throws SQLException {

        if (userId == null) {
            throw new RuntimeException();
        }

        if (species.getNameacademic() == null) {
            throw new IllegalArgumentException("Species academic name must not be null");
        }

        JDBCConnectionJNDI jdbcData = null;

        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "INSERT INTO " + specTable
                    + " (nameacademic, nameen, namees, source, iucnredlist, insertedby) VALUES (?, ?, ?, ?);";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setString(1, species.getNameacademic());
            ps.setString(2, species.getNameen());
            ps.setString(3, species.getNamees());
            ps.setString(4, species.getSource());
            ps.setBoolean(5, species.getIucnredlist());
            ps.setInt(6, userId);
            ps.executeUpdate();

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void createSpeciesTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String specTable = TableTools.getSpeciesTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", specTable)) {
                return;
            }
            System.out.println("Species table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + specTable + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "nameacademic text NOT NULL,"
                    + "nameen text,"
                    + "namees text,"
                    + "iucnredlist boolean,"
                    + "insertedby integer NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
