package de.narimo.georepo.server.api.workspaces;

import java.io.IOException;
import java.sql.SQLException;

import javax.servlet.ServletContext;
import javax.ws.rs.ForbiddenException;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.providers.WorkspaceRegistrationDetails;
import de.narimo.georepo.server.repository.DifftableRepository;
import de.narimo.georepo.server.repository.WorkspaceRepository;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.WorkspacePermissions;

public class WorkspaceService {

    /**
     * NOTE: Usage of this method is discouraged.
     * 
     * It should only be used to set the initial permission for the creator of the
     * workspace.
     * 
     * @param admin
     * @param user
     * @param ws
     * @throws IOException
     * @throws SQLException
     */
    static void setInitialWorkspaceAdminPermission(User user, Workspace ws)
            throws IOException, SQLException {
        setWorkspacePermissions0(ws, user.getName(), WorkspacePermissions.adminPermission);
    }

    /**
     * Set a workspace permission.
     *
     * @param admin
     * @param user
     * @param ws
     * @param permission
     * @throws IOException
     * @throws SQLException
     */
    public static void setWorkspacePermission(User admin, User user, String ws, String permission)
            throws IOException, SQLException {
        AdminTools.checkAdminPermission(admin.getId(), ws);
        Workspace workspace = new Workspace();
        workspace.setName(ws);
        setWorkspacePermissions0(workspace, user.getName(), permission);
    }

    /**
     *
     * All entries from workspaces are cleared for the given combination of userid,
     * workspace and permissions. Then the new entry is written. So there is always
     * only a single unique setting for those combinations.
     *
     * @param workspace
     * @param username
     * @param permission
     *
     * @throws IOException
     * @throws SQLException
     */
    private static void setWorkspacePermissions0(Workspace workspace, String username, String permission)
            throws IOException, SQLException {

        if (workspace == null) {
            throw new IllegalArgumentException("Workspace must not be null;");
        }

        if (!WorkspacePermissions.availablePermissions.contains(permission)) {
            throw new IllegalArgumentException("Unknown workspace permission " + permission);
        }

        Integer userid = UserRepository.getUserId(username);
        if (userid == null) {
            throw new IllegalArgumentException("Cannot set workspace permission for userid " + userid + ";");
        }

        WorkspaceRepository.changeWorkspacePermission(userid, workspace.getName(), permission);
        workspace.setPermission(permission);
    }

    public static Workspace createOrGetDiffWorkspace(ServletContext ctx, Workspace dataWorkspace, User user) {
        AdminTools.checkAdminPermission(user.getId().intValue(), dataWorkspace.getName());

        String dws = DifftableRepository.getDiffWorkspace(dataWorkspace.getName());
        Workspace diffWorkspace = new Workspace(dws);

        // Only create a new diff workspace if none exists for that dataWorkspace!
        // That is, we have to maintain a 1:1 relation between dataWorkspace and
        // diffWorkspace
        if (diffWorkspace.getName() == null) {
            try {
                diffWorkspace = WorkspacesController.createDiffWorkspace(ctx, user, dataWorkspace);
            } catch (IOException e) {
                if (!e.getMessage().contains("Enabling/ Disabling of transaction support")) {
                    throw new RuntimeException(e);
                } else {
                    // This is expected behaviour since the workspace
                    // needs to be activated for WMS/ WFS access.
                    // We don't need it in this case, so we ignore it.
                }
            } catch (SQLException sqle) {
                throw new RuntimeException();
            }
        }
        return diffWorkspace;
    }

    public static void addWorkspaceUser(User user, WorkspaceRegistrationDetails registrationDetails)
            throws SQLException {

        boolean workspaceSecretValid = WorkspaceRepository.isWorkspaceSecretValid(registrationDetails.getWorkspace(),
                registrationDetails.getWorkspaceSecret());

        if (!workspaceSecretValid) {
            throw new ForbiddenException("Registration for workspace failed. Invalid workspace secret.");
        }

        // Add a read-only entry to the workspaces table
        WorkspaceRepository.changeWorkspacePermission(
                user.getId(),
                registrationDetails.getWorkspace(),
                WorkspacePermissions.readPermission);

        // Optionally assign read-write permission for the diff workspace if one
        // exists
        String diffWorkspace = DifftableRepository.getDiffWorkspace(registrationDetails.getWorkspace());
        if (diffWorkspace != null && !diffWorkspace.equals(registrationDetails.getWorkspace())) {
            WorkspaceRepository.changeWorkspacePermission(
                    user.getId(),
                    diffWorkspace,
                    WorkspacePermissions.writePermission);
        }
    }

    public static boolean isRegisteredOnWorkspace(User user, String workspace) {

        String permissionType = WorkspaceRepository.getWorkspacePermissionType(user.getId(), workspace);

        if (WorkspacePermissions.readPermission.contentEquals(permissionType) ||
                WorkspacePermissions.writePermission.contentEquals(permissionType) ||
                WorkspacePermissions.adminPermission.contentEquals(permissionType)) {
            return true;
        }
        return false;
    }
}
