package de.narimo.georepo.server.api.workspaces;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.appconfig.AppKey;
import de.narimo.georepo.server.appconfig.AppSettingsService;
import de.narimo.georepo.server.geoserver.Contact;
import de.narimo.georepo.server.geoserver.GeoserverLayer;
import de.narimo.georepo.server.geoserver.GeoserverWorkspace;
import de.narimo.georepo.server.repository.WorkspaceRepository;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.WorkspacePermissions;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("/workspaces")
@Tag(name = "Workspaces")
public class WorkspacesController {

    AppSettingsService settingsService = new AppSettingsService();

    @GET
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Retrieve workspaces")
    public static Response getWorkspaces(@Context SecurityContext sec, @Context ServletContext ctx) {

        try {
            String username = sec.getUserPrincipal().getName();
            int userId = UserRepository.getUserId(username);

            List<Workspace> workspaces = WorkspaceRepository.getWorkspaces(userId, false);

            Map<String, String> ws = workspaces.stream()
                    .collect(Collectors.toMap(Workspace::getName, Workspace::getPermission));

            return Response.ok().entity(ws).build();
        } catch (Exception e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @POST
    @Path("/")
    @Produces(MediaType.APPLICATION_JSON)
    @Operation(summary = "Creates a new workspace")
    public Response createWorkspace(
            @Context SecurityContext sec,
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            @QueryParam("managed") Boolean createDiffWorkspace,
            @QueryParam("publicAccess") Boolean allowReadOnlyAccessToSystemUser) {

        String username = sec.getUserPrincipal().getName();
        User user = UserRepository.getUser(username).get();
        Workspace dataWorkspace = null;

        try {

            dataWorkspace = createNewDataWorkspace(ctx, user);

            AppKey appkeyFromHeader = AppSettingsService.getAppKeyFromHeader(request);
            if (appkeyFromHeader != null && settingsService.isAppOwner(appkeyFromHeader.getAppKey(), user) &&
                    allowReadOnlyAccessToSystemUser == Boolean.TRUE) {

                Optional<User> optUser = UserRepository
                        .getUser(ctx.getInitParameter("georepo_app_default_readonly_user"));

                if (optUser.isPresent()) {
                    User readonlyUser = optUser.get();
                    // Add a read-only entry to the workspaces table
                    WorkspaceRepository.changeWorkspacePermission(
                            readonlyUser.getId(),
                            dataWorkspace.getName(),
                            WorkspacePermissions.readPermission);
                } else {
                    // do not add workspace permissions for a default user
                }

            }

            if (createDiffWorkspace == Boolean.TRUE) {
                // optionally create diff workspace
                WorkspaceService.createOrGetDiffWorkspace(ctx, dataWorkspace, user);
            }
        } catch (ForbiddenException e) {
            System.out.println("User " + username + " has no permission to create an additional workspace.");
            return Response.status(403).entity("User has no permission to create additional workspace.").build();
        } catch (IOException | SQLException e) {
            System.out.println("An issue occured while creating workspace.");
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
        return Response.status(201).entity(dataWorkspace).build();
    }

    public static Workspace createNewDataWorkspace(ServletContext ctx, User user) throws IOException, SQLException {
        Workspace dataWorkspace = createNewWorkspace0(ctx, user, false);
        WorkspaceRepository.addDataWorkspaceEntry(user, dataWorkspace);
        WorkspaceService.setInitialWorkspaceAdminPermission(user, dataWorkspace);
        return dataWorkspace;
    }

    public static Workspace createDiffWorkspace(ServletContext ctx, User user, Workspace dataWorkspace)
            throws IOException, SQLException {
        Workspace diffWorkspace = createNewWorkspace0(ctx, user, true);
        WorkspaceRepository.addDiffWorkspaceEntry(user, dataWorkspace, diffWorkspace);

        // this adds admin permission for the workspace
        // since the user also has admin permissions for the data workspace
        WorkspaceService.setInitialWorkspaceAdminPermission(user, diffWorkspace);
        diffWorkspace.setPermission(WorkspacePermissions.adminPermission);

        return diffWorkspace;
    }

    /**
     * Creates a workspace with a naming that is not yet taken. Does not yet set
     * permissions on that new workspace.
     * 
     * @param ctx
     * @param user
     * @return
     * @throws IOException
     * @throws SQLException
     */
    public static Workspace createNewWorkspace0(ServletContext ctx, User user, boolean isDiffWorkspace)
            throws IOException, SQLException {

        if (user.getId() == null) {
            throw new IllegalArgumentException("Could not create a workspace for user " + user.getName()
                    + ". Unknown user " + user.getName() + ".");
        }

        if (!isDiffWorkspace) {
            AdminTools.checkCanCreateWorkspace(user);
        } else {
            // do not check against user's workspace count for diff worksapces, since the
            // diff workspace is created automatically
        }

        String newWorkspace = getNewWorkspaceName();
        GeoserverLayer layer = new GeoserverLayer(ctx);
        while (layer.workspaceExists(layer.getWorkspacesXML(), newWorkspace)) {
            // re-create a workspace name which is not yet taken
            newWorkspace = getNewWorkspaceName();
        }

        GeoserverWorkspace ws = new GeoserverWorkspace(newWorkspace, layer.getGeoserverRestUrl(), layer.getHeaders(),
                layer.getGeoserverUser(), layer.getGeoserverPass());
        ws.createWorkspace();

        // add default workspace settings
        Contact georepoContact = new Contact();
        georepoContact.setAddress("Bertolt-Brecht-Allee 24");
        georepoContact.setAddressCity("Dresden");
        georepoContact.setAddressCountry("Germany");
        georepoContact.setAddressPostalCode("01309");
        georepoContact.setAddressType("Postal");
        georepoContact.setAddressElectronicMailAddress("info@georepo.com");
        georepoContact.setContactEmail("info@georepo.com");
        georepoContact.setContactOrganization("narimo systems");
        georepoContact.setContactPosition("Service provider");
        georepoContact.setContactVoice("+49 351 21359935");

        String proxyBaseUrl = ctx.getInitParameter("GEOREPO_URL_EXT") + ctx.getInitParameter("GEOREPO_API_BASE_PATH")
                + "/workspaces";
        ws.modifyWorkspaceSettings("UTF-8", proxyBaseUrl, georepoContact, "https://georepo.com", true);

        // Create WMS and WFS services for that workspace, which must be enabled
        // separately
        ws.createWFSService();
        ws.createWMSService();

        Workspace workspace = new Workspace();
        workspace.setName(newWorkspace);
        workspace.setPermission(WorkspacePermissions.noPermission);
        return workspace;
    }

    public static String getNewWorkspaceName() {
        long workspaceCount = Math.round(Math.random() * 10000);
        return GeorepoConstants.WORKSPACEPREFIX + workspaceCount;
    }
}
