package de.narimo.georepo.server.appconfig;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.api.AppBaseLayersConfiguration;
import de.narimo.georepo.server.api.AppDetailsConfiguration;
import de.narimo.georepo.server.api.app.AppManifestConfiguration;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.ImageAndDocumentService;
import de.narimo.georepo.server.internal.ImageScaleFormat;
import de.narimo.georepo.server.internal.ImageScaleFormats;
import de.narimo.georepo.server.providers.ImageScaler;
import de.narimo.georepo.server.providers.InvoiceGenerator;
import de.narimo.georepo.server.providers.InvoiceGenerator.InvoiceFormat;
import de.narimo.georepo.server.repository.AppSettingsRepository;
import de.narimo.georepo.server.repository.InvoiceRepository;

public class AppSettingsService {

    public List<AppKey> getMyAppKeys(User user) { return AppSettingsRepository.getMyAppKeys(user); }

    public ExtendedAppKey getPublicAppKeyInformation(String appkey, User user) {
        return AppSettingsRepository.getPublicAppKeyInformation(appkey, user);
    }

    public String createAppKey(User user, AppKey appkey) { return AppSettingsRepository.createAppKey(user, appkey); }

    public AppConfiguration getAppConfiguration(String appkey) throws SQLException {
        return AppSettingsRepository.getAppConfiguration(appkey);
    }

    public AppDetailsConfiguration getAppDetailsConfiguration(String appkey) throws NotFoundException, SQLException {
        return AppSettingsRepository.getAppDetailsConfiguration(appkey);
    }

    public AppLayersConfiguration getAppLayersConfiguration(String appkey) throws SQLException {
        return AppSettingsRepository.getAppLayersConfiguration(appkey);
    }

    public AppBaseLayersConfiguration getAppBaseLayersConfiguration(String appkey) throws SQLException {
        return AppSettingsRepository.getAppBaseLayersConfiguration(appkey);
    }

    public List<AppKey> getAppKeysByTemplateKey(String templateKey) {
        return AppSettingsRepository.getAppKeysByTemplateKey(templateKey);
    }

    public AppManifestConfiguration getAppManifest(String appkey) throws SQLException, NotFoundException {
        return AppSettingsRepository.getAppManifestConfiguration(appkey);
    }

    public void saveAppConfiguration(AppConfiguration configuration) throws SQLException {
        AppSettingsRepository.createAppConfigTable();
        AppSettingsRepository.saveAppConfiguration(configuration);
    }

    public void saveAppDetailsConfiguration(AppDetailsConfiguration configuration) throws SQLException {
        AppSettingsRepository.createAppDetailsConfigTable();
        AppSettingsRepository.saveAppDetailsConfiguration(configuration);
    }

    public void saveAppLayersConfiguration(AppLayersConfiguration layersConfiguration) throws SQLException {
        AppSettingsRepository.createAppLayerConfigTable();
        AppSettingsRepository.saveAppLayersConfiguration(layersConfiguration);
    }

    public void saveAppBaseLayersConfiguration(AppBaseLayersConfiguration baseLayersConfiguration) throws SQLException {
        AppSettingsRepository.createAppBaseLayerConfigTable();
        AppSettingsRepository.saveAppBaseLayersConfiguration(baseLayersConfiguration);
    }

    public void saveAppManifest(AppManifestConfiguration appManifestConfiguration) throws SQLException {
        AppSettingsRepository.createAppManifestTable();
        AppSettingsRepository.saveAppManifest(appManifestConfiguration);
    }

    /**
     * CAN WE ACTIVATE AN APP, that has no template key to determine the app
     * provider/ invoice sender???
     * 
     * The invoicing process must be transactional to ensure that no missing id
     * counters persist in the appinvoices table. Reserved counters must be removed
     * from the appinvoices table if the subsequent invoice generation process
     * fails.
     * 
     * @param appkey
     * @param user
     * @throws ForbiddenException
     * @throws IllegalStateException
     */
    public InvoiceGenerator createInvoiceAndActivateAppkey(ServletContext ctx, String appkey, User user)
            throws ForbiddenException, IllegalStateException {

        if (AppSettingsRepository.isAppkeyActivated(appkey, user)) {
            throw new IllegalStateException("The provided appkey is already activated.");
        }

        Map<String, AppKey> invoiceDetails = AppSettingsRepository.getAppkeyInvoiceDetails(appkey, user);

        AppKey appCustomer = invoiceDetails.get("appcustomer");
        AppKey appProvider = invoiceDetails.get("appprovider");

        if (appProvider.getAlias() == null || appProvider.getAlias().trim().equals("")) {
            throw new RuntimeException("The provider appkey does not contain a valid alias.");
        }

        InvoiceGenerator invoicer = new InvoiceGenerator(ctx);
        invoicer.setProductName(appProvider.getAlias());
        invoicer.setTestMode(!ctx.getInitParameter("GEOREPO_CREATE_INVOICES_IN_PRODUCTION_MODE").equals("true"));
        invoicer.setAppCustomer(appCustomer);
        invoicer.setAppProvider(appProvider);

        InvoiceFormat invoiceFormat = InvoiceFormat.german;
        Integer invoiceCounter = null;

        String appInvoicePattern = invoicer.getProductName();

        JDBCConnectionJNDI jdbcAuth = null;
        try {

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            // 0) activate appkey
            // will be deactivated again, when the invoicing process fails
            AppSettingsRepository.activateAppkey(appkey, user);

            // 1) create invoice table
            InvoiceRepository.createInvoicesTable();

            // 2) reserve invoice id
            invoiceCounter = InvoiceRepository.reserveInvoiceId(jdbcAuth, appInvoicePattern, appkey);

            System.out.println("Reserved invoice counter: " + invoiceCounter);

            // 3) create invoice
            Map<String, String> substitutionMap = InvoiceGenerator.createSubstitutionMap(
                    invoiceFormat,
                    appProvider.getOrganisation(),
                    appProvider.getAddressStreet(),
                    appProvider.getAddressStreetnumber(),
                    appProvider.getAddressPostcode(),
                    appProvider.getAddressCity(),

                    appCustomer.getOwnerName(),
                    appCustomer.getAddressStreet(),
                    appCustomer.getAddressStreetnumber(),
                    appCustomer.getAddressPostcode(),
                    appCustomer.getAddressCity(),
                    appCustomer.getAddressCountry());

            invoicer.setSubstitutionMap(substitutionMap);
            invoicer.generateInvoice(invoiceCounter, invoiceFormat);

            // 4) update the invoice name
            InvoiceRepository.updateInvoice(jdbcAuth, invoicer.getInvoiceName(), invoiceCounter, appInvoicePattern);

            System.out.println(
                    "Generating invoice " + invoicer.getInvoiceName() + " was successful and is stored on disk.");

            return invoicer;

        } catch (Exception e) {
            e.printStackTrace();

            try {
                // After failure in the process:
                // Remove invoice id from the appinvoices table
                InvoiceRepository.removeFailedInvoiceId(jdbcAuth, invoiceCounter.intValue(), appInvoicePattern);

                // Remove the invoice from disk after failure
                invoicer.removeFailedInvoiceFromDisk(invoicer.getInvoiceName());
            } catch (SQLException se) {
                se.printStackTrace();
            }

            throw new RuntimeException("Could not create invoice.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public Boolean isAppOwner(String appkey, User user) { return AppSettingsRepository.isAppOwner(appkey, user); }

    public static AppKey getAppKeyFromHeader(HttpServletRequest request) {
        String appkeyFromHeader = request.getHeader("georepo-appkey");
        if (appkeyFromHeader != null) {
            return AppSettingsRepository.getAppKey(appkeyFromHeader);
        }
        return null;
    }

    /**
     * Returns an icon related to an appkey, if it exists.
     * 
     * @param ctx
     * @param appkey
     * @param iconName
     * @return
     * @throws IOException
     */
    public byte[] getAppIcon(ServletContext ctx, String appkey, String iconName) throws IOException {
        ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();
        String extension = imageAndDocumentService.getFileExtension(iconName);

        if (!imageAndDocumentService.isAllowedFileType(extension, DocumentType.IMAGE)) {
            throw new IllegalArgumentException("The file has an incorrect file type: " + extension);
        }

        String georepoServerBasePath = ctx.getInitParameter("GEOREPO_SERVER_BASE_PATH");
        if (georepoServerBasePath == null) {
            throw new RuntimeException("georepo-server base path is not set.");
        }

        String filePath = georepoServerBasePath + "/apps/" + appkey + "/icons/" + iconName;
        System.out.println("Looking up '" + DocumentType.IMAGE.name() + "' (IMAGE or DOCUMENT): " + filePath);

        if (!Files.exists(new File(filePath).toPath(), LinkOption.NOFOLLOW_LINKS)) {
            throw new NotFoundException();
        }
        return Files.readAllBytes(new File(filePath).toPath());

    }

    public void addAppIcons(ServletContext ctx, InputStream uploadedInputStream, String appkey, String iconName) {
        ImageAndDocumentService imageAndDocumentService = new ImageAndDocumentService();
        ImageScaleFormats scaleFormats = getScaleFormats();
        BufferedImage bi = null;
        try {

            String georepoServerBasePath = ctx.getInitParameter("GEOREPO_SERVER_BASE_PATH");
            if (georepoServerBasePath == null) {
                throw new RuntimeException("georepo-server base path is not set.");
            }

            // create the base directory, if not exists
            File basePath = new File(georepoServerBasePath + "/apps");
            if (basePath.exists() && !basePath.isDirectory()) {
                throw new RuntimeException("App directory must be a directory.");
            }

            // add the image to the filesystem
            String imgDir = basePath + "/" + appkey + "/icons";
            File imagePath = new File(imgDir);
            if (!imagePath.exists()) {
                imagePath.mkdirs();

            } else if (imagePath.isDirectory()) {
                if (imagePath.list().length > 0) {
                    // Remove all previous icons so that they might not appear cached, if we changed
                    // the available icon names
                    try (DirectoryStream directoryStream = Files.newDirectoryStream(imagePath.toPath())) {
                        Iterator<Path> iterator = directoryStream.iterator();
                        while (iterator.hasNext()) {
                            Files.delete(iterator.next());
                        }
                    } catch (Exception e) {
                        throw new InternalError("Could not read the template app icon directoy.");
                    }
                }
            }

            bi = ImageIO.read(uploadedInputStream);

            for (ImageScaleFormat format : scaleFormats.getImageScaleFormats()) {
                if (format != null) {
                    // protect ratio option:
//                    if (format.get().size() < 1) {
//                        targetHeight = 100;
//                        float ratio = (float) bi.getWidth() / bi.getHeight();
//                        targetWidth = Math.round(targetHeight * ratio);
//                    } else {
//                    int targetHeight = format.getHeight();
//                    int targetWidth = format.getOutputFormats().get(0)[1];
//                    }
                    BufferedImage resizedImage = resizeImage(bi, format.getWidth(), format.getHeight());

                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ImageIO.write(resizedImage, "png", baos);

                    saveAppIcon(ctx, new ByteArrayInputStream(baos.toByteArray()), appkey, format.getName(),
                            imageAndDocumentService.getFileExtension(iconName), imgDir);
                }
            }
        } catch (IOException e) {
            throw new InternalError("Could not resize image.", e);
        }
    }

    /**
     * Save an image to file system
     * 
     * @throws IOException
     */
    private void saveAppIcon(ServletContext ctx, InputStream fileInputStream, String appkey, String iconName,
            String fileExtension, String iconDirPath) throws IOException {

        // do checks on image
        if (!isAllowedAppIconFileType(fileExtension)) {
            throw new IllegalArgumentException("Unsupported filetype " + fileExtension
//                        + ". Must be image/png, image/jpg, image/jpeg or image/gif.");
                    + ". Must be image/png");
        }

        String[] iconNameParts = iconName.split("\\.");
        String individualIconName = iconNameParts[0] + "-" + appkey + "." + iconNameParts[1];
        File outFile = new File(iconDirPath + "/" + individualIconName);

        System.out.println("Filepath for new file: " + outFile.toString());

//        if (documentType.equals(DocumentType.IMAGE)) {
        // TODO: check image dimensions or file size
//            BufferedImage bufferedImage = ImageIO.read(fileInputStream);
//            if (bufferedImage.getWidth() > 1920 || bufferedImage.getHeight() > 1080) {
//                throw new IOException("Image dimensions should not exceed 1920x1080 pixels.");
//            }
//            ImageIO.write(bufferedImage, fileExtension, outFile);
//        }
        Files.copy(fileInputStream, outFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    /**
     * Create default icons for an app.
     * 
     * @param ctx
     * @param appkey
     * @throws IOException
     * @throws FileNotFoundException
     */
    public void createDefaultAppIcons(ServletContext ctx, String appkey) throws FileNotFoundException, IOException {
        String templateIconDir = ctx.getInitParameter("GEOREPO_DEFAULT_ICON_PATH");
        if (templateIconDir == null) {
            throw new RuntimeException("Default app icon path not found.");
        }
        File templateIconDirPath = new File(templateIconDir);

        try (InputStream iconStream = new FileInputStream(templateIconDirPath)) {
            addAppIcons(ctx, iconStream, appkey, templateIconDirPath.getName());
        }
    }

    private static BufferedImage resizeImage(BufferedImage originalImage, int targetWidth, int targetHeight)
            throws IOException {
        return ImageScaler.resizeImage(originalImage, targetWidth, targetHeight);
    }

    private static ImageScaleFormats getScaleFormats() {
        List<ImageScaleFormat> scaleFormats = new ArrayList<>();
        scaleFormats.add(new ImageScaleFormat(128, 128, "icon-128x128.png"));
        scaleFormats.add(new ImageScaleFormat(192, 192, "icon-192x192.png"));
        scaleFormats.add(new ImageScaleFormat(256, 256, "icon-256x256.png"));
        scaleFormats.add(new ImageScaleFormat(384, 384, "icon-384x384.png"));
        scaleFormats.add(new ImageScaleFormat(512, 512, "icon-512x512.png"));
        scaleFormats.add(new ImageScaleFormat(16, 16, "favicon.ico"));
        scaleFormats.add(new ImageScaleFormat(16, 16, "favicon-16x16.png"));
        scaleFormats.add(new ImageScaleFormat(32, 32, "favicon-32x32.png"));
        scaleFormats.add(new ImageScaleFormat(96, 96, "favicon-96x96.png"));
        scaleFormats.add(new ImageScaleFormat(128, 128, "favicon-128x128.png"));
        scaleFormats.add(new ImageScaleFormat(192, 192, "favicon-192x192.png"));
        scaleFormats.add(new ImageScaleFormat(256, 256, "favicon-256x256.png"));
        scaleFormats.add(new ImageScaleFormat(384, 384, "favicon-384x384.png"));
        scaleFormats.add(new ImageScaleFormat(512, 512, "favicon-512x512.png"));
        scaleFormats.add(new ImageScaleFormat(120, 120, "apple-icon-120x120.png"));
        scaleFormats.add(new ImageScaleFormat(152, 152, "apple-icon-152x152.png"));
        scaleFormats.add(new ImageScaleFormat(167, 167, "apple-icon-167x167.png"));
        scaleFormats.add(new ImageScaleFormat(180, 180, "apple-icon-180x180.png"));
        scaleFormats.add(new ImageScaleFormat(1125, 2436, "apple-launch-1125x2436.png"));
        scaleFormats.add(new ImageScaleFormat(1170, 2532, "apple-launch-1170x2532.png"));
        scaleFormats.add(new ImageScaleFormat(1242, 2208, "apple-launch-1242x2208.png"));
        scaleFormats.add(new ImageScaleFormat(1242, 2688, "apple-launch-1242x2688.png"));
        scaleFormats.add(new ImageScaleFormat(1284, 2778, "apple-launch-1284x2778.png"));
        scaleFormats.add(new ImageScaleFormat(1536, 2048, "apple-launch-1536x2048.png"));
        scaleFormats.add(new ImageScaleFormat(1620, 2160, "apple-launch-1620x2160.png"));
        scaleFormats.add(new ImageScaleFormat(1668, 2224, "apple-launch-1668x2224.png"));
        scaleFormats.add(new ImageScaleFormat(1668, 2388, "apple-launch-1668x2388.png"));
        scaleFormats.add(new ImageScaleFormat(2048, 2732, "apple-launch-2048x2732.png"));
        scaleFormats.add(new ImageScaleFormat(750, 1334, "apple-launch-750x1334.png"));
        scaleFormats.add(new ImageScaleFormat(828, 1792, "apple-launch-828x1792.png"));

        scaleFormats.add(new ImageScaleFormat(144, 144, "ms-icon-144x144.png"));
//        scaleFormats.add(new ImageScaleFormat(128, 128, "safari-pinned-tab.svg"));

        ImageScaleFormats imageScaleFormats = new ImageScaleFormats();
        imageScaleFormats.setImageScaleFormats(scaleFormats);
        return imageScaleFormats;
    }

    public boolean isAllowedAppIconFileType(String fileType) {
        if (fileType == null) {
            return false;
        }
        return "image/png".equals("image/" + fileType.toLowerCase());
    }

    public static void main(String[] args) throws IOException {

//        String img = "/home/ulrich/Documents/projects/mhs/georepo-atlas/logo.png";
        String img = "/home/ulrich/temp/uma/logo_gernsbach.png";
        File f = new File(img);
        FileInputStream fis = new FileInputStream(f);

//        ImageScaler scaler = new ImageScaler(fis);
//        scaler.run();

//        File outputfile = new File("/home/ulrich/tmp/georepo-icon-2048x2732-copy2.png");
//        ImageIO.write(resizeImage(ImageIO.read(f), 256, 256), "png", outputfile);

        // ---------------

        try {
            BufferedImage bi = ImageIO.read(fis);

            ImageScaleFormat format = getScaleFormats().getImageScaleFormats().get(0);

            BufferedImage resizedImage = resizeImage(bi, format.getWidth(), format.getHeight());

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(resizedImage, "png", baos);

            File imagePath = new File("/home/ulrich/temp/uma/out.png");

            File outFile = imagePath; // new File(imagePath + "/" + iconName);
            System.out.println("Filepath for new file: " + outFile.toString());

            Files.copy(new ByteArrayInputStream(baos.toByteArray()), outFile.toPath(),
                    StandardCopyOption.REPLACE_EXISTING);

        } catch (Exception e) {
            System.out.println("ERROR");
        }
    }
}
