package de.narimo.georepo.server.filter;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Priority;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.http.URLResponse;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.commons.json.JsonConverter;
import de.narimo.commons.ws.http.HttpMethod;
import de.narimo.commons.ws.http.HttpURLClient;
import de.narimo.georepo.server.db.GeolocationEntry;
import de.narimo.georepo.server.repository.ParentDatasetRepository;

@Provider
@Priority(Priorities.USER)
public class GeoLoggingFilter implements ContainerRequestFilter {

    @Context
    private HttpServletRequest httpServletRequest;

    @Context
    private UriInfo uriInfo;

    @Override
    public void filter(ContainerRequestContext crc) throws IOException {
        try {
            // only do this, if the request is a startup request for the app which
            // contains an appkey
            // /georepo-server/api/apps/A2SlJRH07KnMaqtjeOhzfbrkfgKydCJm/details?
            String ip = getOriginalClientIP(httpServletRequest);
            if (ip == null) {
                return;
            }
            if (httpServletRequest.getMethod().equals("OPTIONS")) {
                return;
            }

            String url = uriInfo.getRequestUri().toString();
            String appkey = getAppkey(httpServletRequest);

            if (isRelevantRequest(httpServletRequest)) {
                // separate task to not interfere with request run-time
                Runnable r = new Runnable() {
                    @Override
                    public void run() {
                        try {
                            createGeoIPTable();
                            if (appkey == null) {
                                addGeoEntry(null, ip, null, null, url);
                            } else if (storeGeoLocationAllowed(appkey)) {
                                GeolocationEntry ge = getRecentIPEntry(ip, appkey);
                                if (ge != null) {
                                    addGeoEntry(appkey, ip, ge.getCountry(), ge.getRegion(), url);
                                } else {
                                    // get the country and region for the ip
                                    Map<String, String> geolocation = getGeoLocation(ip);
                                    addGeoEntry(appkey, ip, geolocation.get("country"), geolocation.get("region"), url);
                                }
                            } else if (appkey != null) {
                                addGeoEntry(appkey, ip, null, null, url);
                            }
                            obscureOldIPEntries();
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                };
                new Thread(r).start();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void createGeoIPTable() {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", "geoaccesslog")) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public.geoaccesslog"
                    + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "appkey text,"
                    + "ip text,"
                    + "country text,"
                    + "region text,"
                    + "accesstime timestamp without time zone DEFAULT timezone('utc'::text, now()),"
                    + "url text"
                    + ");";
            jdbcData.execute(sql);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    private static void addGeoEntry(String appkey, String ip, String country, String region, String url) {
        JDBCConnectionJNDI jdbcData = null;
        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String query = "INSERT INTO public.geoaccesslog (appkey, ip, country, region, url) VALUES (?, ?, ?, ?, ?);";
            PreparedStatement ps = jdbcData.prepareStatement(query);
            ps.setString(1, appkey);
            ps.setString(2, ip);
            ps.setString(3, country);
            ps.setString(4, region);
            ps.setString(5, url);
            ps.execute();
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
            }
        }
    }

    /**
     * Checks, whether there already exists an entry for the same ip and appkey
     * within the last x hours. In this case, we won't attempt to resolve that IP
     * again to stay within the geoip resolution limit.
     * 
     * @param appkey
     * @param ip
     * @param country
     * @param region
     * @param url
     */
    private static GeolocationEntry getRecentIPEntry(String ip, String appkey) {
        JDBCConnectionJNDI jdbcData = null;
        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String query = "SELECT country, region FROM public.geoaccesslog WHERE ip = ? AND appkey = ? AND accesstime > now() - interval '5 hours';";
            PreparedStatement ps = jdbcData.prepareStatement(query);
            ps.setString(1, ip);
            ps.setString(2, appkey);
            ResultSet rs = ps.executeQuery();
            if (!rs.next()) {
                return null;
            }
            String country = rs.getString("country");
            String region = rs.getString("region");
            GeolocationEntry ge = new GeolocationEntry();
            ge.setCountry(country);
            ge.setRegion(region);
            return ge;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
            }
        }
    }

    public static String getOriginalClientIP(HttpServletRequest httpServletRequest) {
        String ip = httpServletRequest.getRemoteAddr();
        try {
            Enumeration<String> headers = httpServletRequest.getHeaderNames();
            while (headers.hasMoreElements()) {
                String hname = headers.nextElement();
                System.out.println("forwarded: " + hname + " - " + httpServletRequest.getHeader(hname));
            }

            // intermediate proxy ips
            String ips = httpServletRequest.getHeader("X-Forwarded-For");
            // original client ip
            if (ips != null && ips.length() > 0) {
                ip = ips.split(",")[0];
            }
        } catch (Exception ignored) {
            ignored.printStackTrace();
            throw new RuntimeException();
        }

        if (ip == null) {
            ip = httpServletRequest.getRemoteAddr();
        }
        return ip;
    }

    /**
     * Obscure ip entries that are older than x hours.
     */
    private static void obscureOldIPEntries() {
        JDBCConnectionJNDI jdbcData = null;
        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String query = "UPDATE geoaccesslog SET ip = ("
                    + "SELECT REPLACE (ip, substr(ip, 0, char_length(ip)-(strpos(reverse(ip), '.'))+2), 'x.x.x.' )"
                    + ") where accesstime < now() - interval '5 hours'"
                    + " AND ascii(ip) <> 120";
            PreparedStatement ps = jdbcData.prepareStatement(query);
            ps.execute();
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
            }
        }
    }

    /**
     * Checks, whether storing the geolocation for an appkey is allowed.
     * 
     * @param appkey
     * @return
     */
    private static boolean storeGeoLocationAllowed(String appkey) {
        if (appkey == null || appkey.equals("")) {
            return false;
        }

        JDBCConnectionJNDI jdbcData = null;
        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String query = "SELECT storegeoip from public.appkeys where appkey = ?;";
            PreparedStatement ps = jdbcData.prepareStatement(query);
            ps.setString(1, appkey);
            ResultSet rs = ps.executeQuery();
            rs.next();
            return rs.getBoolean("storegeoip");
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
            }
        }
    }

    private static boolean isRelevantRequest(HttpServletRequest httpServletRequest) {
        String method = httpServletRequest.getMethod();
        String url = httpServletRequest.getRequestURI();
        if (method.equals("GET") && url.contains("/api/apps/") && url.contains("/details")) {
            return true;
        }
        return false;
    }

    private static String getAppkey(HttpServletRequest httpServletRequest) {
        String url = httpServletRequest.getRequestURI();
        String appkey = null;
        // 1) check query params for appkey
//        String appkey = httpServletRequest.getParameter("appkey");
        // 2) check appkey within the url
//        if (appkey == null && url.contains("/#/")) {
//            // appkey could be in the request url as well
//            appkey = url.substring(url.indexOf("/#/") + 3);
//        } else 
        // 3) check the /details endpoint for an appkey
        if (appkey == null && url.contains("/api/apps/") && url.contains("/details")) {
            appkey = url.substring(url.indexOf("/api/apps") + 10);
            appkey = appkey.substring(0, appkey.indexOf("/details"));
        }
        return appkey;
    }

    /**
     * Returns the geolocation country and region resolved by third-party provider
     * or null for both, if the given ip is incorrect.
     * 
     * @param ip
     * @return Map<String, String> with the keys country and region
     * @throws IOException
     */
    private static Map<String, String> getGeoLocation(String ip) throws IOException {
        Map<String, String> geolocation = new HashMap<>();
        geolocation.put("country", "");
        geolocation.put("region", "");

        if (ip != null) {
            int countDelimiters = ip.replace(".", "--").length() - ip.length();
            if (countDelimiters == 3) {
                String ipServerUrl = "https://api.db-ip.com/v2/free/" + ip;
                URLResponse rr = HttpURLClient.sendRequest(ipServerUrl, null, null, null, HttpMethod.GET, null, null,
                        null,
                        true, false);

                GeoLocationDBIPReponse geolocationResponse = JsonConverter.fromJson(rr.getResponseBody(),
                        GeoLocationDBIPReponse.class);
                geolocation.put("country", geolocationResponse.getCountryName());
                geolocation.put("region", geolocationResponse.getCity());
            }
        }
        return geolocation;
    }
}
