package de.narimo.georepo.server.noauth;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.ext.Provider;

import de.narimo.commons.dto.RegistrationLanguage;
import de.narimo.commons.dto.User;
import de.narimo.geocore.ws.registration.RegistrationProvider;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.appconfig.AppKey;
import de.narimo.georepo.server.appconfig.AppSettingsService;
import de.narimo.georepo.server.notification.Notifier;
import io.swagger.v3.oas.annotations.tags.Tag;

@Provider
@Path("")
@Tag(name = "RegistrationController")
public class RegistrationController {

    @POST
    @Path("/register")
    @Consumes(MediaType.APPLICATION_JSON)
    public static Response registerNewUser(
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            UserRegistrationDetails registrationDetails) {

        try {
            RegistrationCheck.doCrawlerCheck(request.getHeader("user-agent"));

            RegistrationLanguage userLanguage = RegistrationLanguage
                    .getRegistrationLanguage(registrationDetails.getLanguage());

            RegistrationCheck.doEmailCheck(registrationDetails.getEmail());

            String appAlias = null;
            String appOwnerEmail = null;
            RegistrationLanguage appOwnerLanguage = null;

            AppKey appkeyFromHeader = AppSettingsService.getAppKeyFromHeader(request);
            if (appkeyFromHeader != null) {
                appAlias = appkeyFromHeader.getAlias();
                User appOwner = UserRepository.getUsersById(Arrays.asList(appkeyFromHeader.getCreatedBy())).get(0);
                if (appOwner == null) {
                    // what to do, if the app owner has been deactivated or is not found?
                    // is this a relevant use case?
                    // right now, we just don't send a notification to the app owner
                } else {
                    appOwnerEmail = appOwner.getEmail();
                    appOwnerLanguage = appOwner.getLanguage();
                }

                // 2.8: set the client app from the appkey, not as sent by the client
                registrationDetails.setClientApp(appAlias);
            }

            String secret = RegistrationProvider.registerUser(registrationDetails);

            String georepoPlatform = ctx.getInitParameter("GEOREPO_URL_EXT");
            String registrationLink = georepoPlatform + "/georepo-server/registration/confirm?"
                    + "m=" + URLEncoder.encode(registrationDetails.getEmail(), StandardCharsets.UTF_8.toString())
                    + "&s=" + secret;

            Notifier notifier = Notifier.getInstance(ctx, appkeyFromHeader);
            notifier.notifyUserAboutRegistrationConfirmationLink(registrationDetails, appAlias, registrationLink,
                    userLanguage);
            notifier.notifyNarimoAboutRegistration(registrationDetails, appAlias, RegistrationLanguage.en_US);
            notifier.notifyAppOwnerAboutUserRegistration(registrationDetails, appAlias,
                    appOwnerEmail, appOwnerLanguage);

            return Response.ok().entity("Registration successful but needs confirmation.").build();

        } catch (IOException e) {
            e.printStackTrace();
            if (e.getMessage().contains("Username or email is already registered")) {
                return Response.status(Status.CONFLICT).entity("The email is already registered.").build();
            }
            return Response.status(Status.BAD_REQUEST).build();
        } catch (NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    @GET
    @Path("/confirm")
    public static Response confirmNewUser(
            @Context ServletContext ctx,
            @Context HttpServletRequest request,
            @QueryParam("m") String email,
            @QueryParam("s") String secret) {

        try {
            RegistrationCheck.doCrawlerCheck(request.getHeader("user-agent"));

            if (email == null || secret == null) {
                throw new IOException("Invalid confirmation link.");
            }

            RegistrationProvider.confirmRegistration(email, secret);

            String confirmation = "<html><center><div style='color:#224366;margin:40px'>Your registration has been successfully confirmed.</div></center></html>";
            return Response.ok().entity(confirmation).build();

        } catch (IOException e) {
            e.printStackTrace();
            return Response.status(Status.BAD_REQUEST).build();
        } catch (NotFoundException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (InternalError | RuntimeException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }
}
