package de.narimo.georepo.server.notification;

import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.List;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.servlet.ServletContext;

import de.narimo.commons.dto.RegistrationLanguage;
import de.narimo.commons.dto.User;
import de.narimo.commons.ws.mail.Mailer;
import de.narimo.geocore.ws.registration.PasswordForgottenDetails;
import de.narimo.geocore.ws.registration.UserRegistrationDetails;
import de.narimo.georepo.server.appconfig.AppKey;
import de.narimo.georepo.server.tools.AESGenerator;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.Translator;

public class Notifier {

    private Translator t = new Translator();
    private Mailer mailer;
    private String mailerUser;
    private String mailerPass;
    private String sender;
    private String defaultApplication;

    public Notifier(ServletContext ctx) {
        String mailHost = ctx.getInitParameter("info_mail_host");
        Integer mailPort = Integer.valueOf(ctx.getInitParameter("info_mail_port"));
        String mailProtocol = ctx.getInitParameter("info_mail_protocol");

        mailer = new Mailer(mailHost, mailPort, mailProtocol);
        mailerUser = ctx.getInitParameter("info_mail_user");
        mailerPass = ctx.getInitParameter("info_mail_pw");
        sender = ctx.getInitParameter("info_mail_sender");
        defaultApplication = ctx.getInitParameter("info_mail_default_application_name");
    }

    public Notifier(ServletContext ctx, Mailer mailer, String mailingUser, String aesEncryptedMailingPass) {
        this.mailer = mailer;
        this.mailerUser = mailingUser;
        try {
            this.mailerPass = new AESGenerator(ctx).decrypt(aesEncryptedMailingPass);
        } catch (InvalidKeyException | NoSuchAlgorithmException | InvalidKeySpecException | NoSuchPaddingException
                | InvalidAlgorithmParameterException | BadPaddingException | IllegalBlockSizeException e) {
            throw new RuntimeException("AES decrypt error.");
        }
        sender = mailingUser;
        defaultApplication = ctx.getInitParameter("info_mail_default_application_name");
    }

    /**
     * Returns a notifier for a specific appkey's mailing settings or the default
     * georepo mailing settings.
     * 
     * @param ctx
     * @param appkey
     * @return
     */
    public static Notifier getInstance(ServletContext ctx, AppKey appkey) {
        if (appkey != null && appkey.getMailingHost() != null && appkey.getMailingUser() != null
                && appkey.getMailingPw() != null) {
            // use a mailer with app specific mail address and default smtp port
            Mailer mailer = new Mailer(appkey.getMailingHost(), null, null);
            return new Notifier(ctx, mailer, appkey.getMailingUser(),
                    appkey.getMailingPw());
        } else {
            // use default georepo mailer
            return new Notifier(ctx);
        }
    }

    /**
     * Send notification to workspace admins that POI changes have been submitted.
     */
    public void notifyWorkspaceAdminsAboutPOIChanges(String workspace, String appName,
            String dataLayerName) {

        Thread thread = new Thread("notifyAdminsAboutPOIChanges") {
            @Override
            public void run() {
                List<User> admins = AdminTools.getAdminUsers(workspace);
                for (User admin : admins) {
                    sendPoiUpdateAdminNotification(dataLayerName, admin.getEmail(), appName, admin.getLanguage());
                }
            }
        };
        thread.start();
    }

    /**
     * Send notification to workspace admins that a new user has requested
     * registration.
     */
    public void notifyWorkspaceAdminsAboutRegistration(String workspace, String appName,
            String userEmail) {

        Thread thread = new Thread("notifyAdminsAboutRegistration") {
            @Override
            public void run() {
                List<User> admins = AdminTools.getAdminUsers(workspace);
                for (User admin : admins) {
                    notifyRecipientAboutWorkspaceRegistration(workspace, userEmail,
                            admin.getEmail(), appName, admin.getLanguage());
                }
            }
        };
        thread.start();
    }

    /**
     * Notify a user about his successful registration.
     */
    public void notifyUserAboutRegistrationConfirmationLink(UserRegistrationDetails registrationDetails, String appName,
            String registrationLink, RegistrationLanguage language) {

        Thread thread = new Thread("notifyRegistration") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_NEW_USER_REGISTRATION_SUBJECT", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_NEW_USER_REGISTRATION", language, registrationLink);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);

                mailer.sendMail(sender, registrationDetails.getEmail(), subject, msgBody + msgTeam, mailerUser,
                        mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Notify narimo that a registration has been requested.
     *
     * @param ctx
     * @param registrationDetails
     */
    public void notifyNarimoAboutRegistration(UserRegistrationDetails registrationDetails, String appName,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyRecipientAboutRegistration") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_NEW_USER_REGISTRATION_NARIMO_SUBJECT", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_NEW_USER_REGISTRATION_NARIMO", language,
                        registrationDetails.getEmail());
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Notify an app owner about a new registration request for his application.
     * 
     * @param registrationDetails
     * @param language
     */
    public void notifyAppOwnerAboutUserRegistration(UserRegistrationDetails registrationDetails, String appName,
            String ownerEmail, RegistrationLanguage language) {

        if (ownerEmail != null) {
            Thread thread = new Thread("notifyRecipientAboutRegistration") {
                @Override
                public void run() {
                    String subject = t.translate("TRANSLATION_NEW_USER_REGISTRATION_OWNER_SUBJECT", language,
                            appName);
                    String msgBody = t.translate("TRANSLATION_NEW_USER_REGISTRATION_OWNER", language,
                            registrationDetails.getEmail());
                    String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, defaultApplication);

                    mailer.sendMail(sender, ownerEmail, subject, msgBody + msgTeam, mailerUser, mailerPass);
                }
            };
            thread.start();
        } else {
            System.out.println("Could not send notification to '" + appName + "'owner, because ownerEmail is null.");
        }
    }

    private void notifyRecipientAboutWorkspaceRegistration(
            String workspace, String registeredUserEmail, String recipientEmail, String appName,
            RegistrationLanguage language) {

        appName = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

        String subject = t.translate("TRANSLATION_NEW_USER_WORKSPACE_REGISTRATION_SUBJECT", language);
        String msgBody = t.translate("TRANSLATION_NEW_USER_WORKSPACE_REGISTRATION", language, recipientEmail,
                registeredUserEmail);
        String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appName);

        mailer.sendMail(sender, recipientEmail, subject, msgBody + msgTeam, mailerUser, mailerPass);
    }

    /**
     * Send password reset mail when a password forgotten request was submitted.
     */
    public void notifyUserAboutPasswordResetLink(PasswordForgottenDetails passwordDetails, String appName,
            String resetLink, RegistrationLanguage language) {

        Thread thread = new Thread("notifyPasswordForgotten") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_PASSWORD_RESET_REQUEST_SUBJECT", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_PASSWORD_RESET_REQUEST", language, resetLink);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);

                mailer.sendMail(sender, passwordDetails.getEmail(), subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyUserAboutSuccessfulPasswordReset(String recipientMail, String appName,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyPasswordResetSuccessful") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_PASSWORD_RESET_SUBJECT", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_PASSWORD_RESET", language);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);

                mailer.sendMail(sender, recipientMail, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyUserAboutAccountDeletion(String recipientMail, String appName,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyAccountDeleteConfirmation") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_ACCOUNT_DELETION_USER_HEADER", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_ACCOUNT_DELETION_USER", language);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);

                mailer.sendMail(sender, recipientMail, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyNarimoAboutAccountRemoval(String userAccountMail, String appName,
            RegistrationLanguage language) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_ACCOUNT_DELETION_HEADER", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_ACCOUNT_DELETION", language,
                        userAccountMail);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);

                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void notifyNarimoAboutFailedAccountRemoval(String userAccountMail,
            String appName, RegistrationLanguage language) {

        Thread thread = new Thread("notifyNarimoAboutAccountDeletion") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_FAILED_ACCOUNT_DELETION_HEADER", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_FAILED_ACCOUNT_DELETION", language,
                        userAccountMail);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);
                mailer.sendMail(sender, "info@narimo.de", subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    /**
     * Send a notification that POI changes have been submitted.
     *
     * @param ctx
     * @param dataLayerName
     * @param recipient
     */
    private void sendPoiUpdateAdminNotification(String dataLayerName, String recipient, String appName,
            RegistrationLanguage language) {

        String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

        String subject = t.translate("TRANSLATION_SUGGESTED_DATASET_UPDATE_HEADER", language,
                appNamex);
        String msgBody = t.translate("TRANSLATION_SUGGESTED_DATASET_UPDATE", language, dataLayerName);
        String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);
        mailer.sendMail(sender, recipient, subject, msgBody + msgTeam, mailerUser, mailerPass);
    }

    public void sendDeclinedPoiUpdateNotification(String dataLayerName, String recipient, String appName,
            RegistrationLanguage language) {
        Thread thread = new Thread("updateDeclined") {
            @Override
            public void run() {
                String appNamex = (appName != null && !appName.isEmpty()) ? appName : defaultApplication;

                String subject = t.translate("TRANSLATION_DATASET_UPDATE_DECLINED_HEADER", language,
                        appNamex);
                String msgBody = t.translate("TRANSLATION_DATASET_UPDATE_DECLINED", language,
                        dataLayerName);
                String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, appNamex);
                mailer.sendMail(sender, recipient, subject, msgBody + msgTeam, mailerUser, mailerPass);
            }
        };
        thread.start();
    }

    public void sendGeorepoInvoice(String recipient, String customerAppAlias, String providerAppAlias,
            RegistrationLanguage language, String invoiceName, String customerName, String attachmentPath) {

        if (recipient == null)
            throw new IllegalArgumentException("Recipient may not be null.");
        if (customerAppAlias == null)
            throw new IllegalArgumentException("Customer app alias may not be null.");
        if (providerAppAlias == null)
            throw new IllegalArgumentException("Provider app alias may not be null.");
        if (customerName == null)
            throw new IllegalArgumentException("Customer name may not be null.");
        if (invoiceName == null)
            throw new IllegalArgumentException("Invoice name may not be null.");
        if (attachmentPath == null)
            throw new IllegalArgumentException("Attachment path may not be null.");

        String subject = t.translate("TRANSLATION_NEW_INVOICE_SUBJECT", language, customerAppAlias,
                invoiceName);
        String msgBody = t.translate("TRANSLATION_NEW_INVOICE", language, customerName, customerAppAlias);
        String msgTeam = t.translate("TRANSLATION_TEAM_SIGNATURE", language, providerAppAlias);

        mailer.sendMailWithAttachment(sender, recipient, subject, msgBody + msgTeam, mailerUser,
                mailerPass, attachmentPath);
    }

    public static void main(String[] args) throws IOException, InvalidKeyException, NoSuchPaddingException,
            InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException,
            NoSuchAlgorithmException, InvalidKeySpecException {

//        Translator t = new Translator();
//        System.out.println(
//                t.translate("TRANSLATION_SUGGESTED_DATASET_UPDATE_HEADER", RegistrationLanguage.de_DE, "4", "some"));

    }
}