package de.narimo.georepo.server.providers;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Year;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletContext;

import org.jsoup.Jsoup;
import org.jsoup.helper.W3CDom;
import org.jsoup.nodes.Document;

import com.openhtmltopdf.pdfboxout.PdfRendererBuilder;

import de.narimo.georepo.server.appconfig.AppKey;

public class InvoiceGenerator {

    private boolean debugMode = false;
    private String altInvoicingBasepath = "";
    private Map<String, String> substitutionMap;
    private String productName;
    private boolean testMode = true;
    private ServletContext servletContext;

    private boolean invoiceAlreadySent = false;

    // params which can be retrieved after having created the invoice
    private String invoiceName;
    private int invoiceYear;
    private int invoiceMonth;
    private String invoicePath;
    private AppKey appCustomer;
    private AppKey appProvider;

    // after the product name has been requested on this instance, we may not
    // change it anymore!
    private boolean productNameIsFixed = false;

    public enum InvoiceFormat {
        german,
        international
    }

    public InvoiceGenerator(ServletContext servletContext) { this.servletContext = servletContext; }

    public String generateInvoice(Integer invoiceCounter, InvoiceFormat invFormat) {

        if (invoiceAlreadySent) {
            throw new IllegalStateException(
                    "This InvoiceGenerator has already been used to send an invoice. Please create a new instance.");
        }

        productNameIsFixed = true;

        String GEOREPO_INVOICING_BASEPATH = null;
        if (servletContext != null) {
            String GEOREPO_SERVER_BASE_PATH = servletContext.getInitParameter("GEOREPO_SERVER_BASE_PATH");
            if (GEOREPO_SERVER_BASE_PATH == null) {
                throw new RuntimeException();
            }
            GEOREPO_INVOICING_BASEPATH = GEOREPO_SERVER_BASE_PATH + "/invoices";
        } else if (altInvoicingBasepath != null) {
            GEOREPO_INVOICING_BASEPATH = altInvoicingBasepath;
        } else {
            throw new RuntimeException("Missing servlet context or invoicing basepath.");
        }

        invoiceYear = Year.now().getValue();
        String year = String.valueOf(invoiceYear);
        invoiceMonth = Calendar.getInstance().get(Calendar.MONTH) + 1;
        String month = String.format("%02d", invoiceMonth);

        String counterFormat = "%03d";
        String counter = String.format(counterFormat, invoiceCounter);

        String invoicePrefix = invFormat == InvoiceFormat.german ? "RE" : "INV";
        this.invoiceName = invoicePrefix + "-" + retrieveFinalProductName() + "-" + year + month + counter;

        String outputDir = GEOREPO_INVOICING_BASEPATH + "/" + year + "/";
        String intermediateOutputHtml = outputDir + invoiceName + ".html";
        this.invoicePath = outputDir + invoiceName + ".pdf";

        if (!this.debugMode) {
            File outputPdfFile = new File(invoicePath);
            if (outputPdfFile.exists() && !outputPdfFile.isDirectory()) {
                throw new IllegalArgumentException(
                        "The invoice " + invoiceName
                                + " to generate already exists. Please increase the invoice counter.");
            }
        }
        new File(outputDir).mkdirs();

        String htmlTemplate = GEOREPO_INVOICING_BASEPATH + "/invoice-template.html";

        String content = "";
        try {
            // read the html template file and subsitute placeholders
            content = readFile(htmlTemplate, StandardCharsets.UTF_8);
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException("Creating the invoice failed.");
        }
        // escape % signs in html
        content = content.replace("%", "%%");

        if (substitutionMap == null) {
            throw new IllegalArgumentException("You need to provide the substitution map.");
        }

        // override a few entries
        substitutionMap.put("invoicename", invoiceName);
        if (testMode) {
            substitutionMap.put("invoicehead", substitutionMap.get("invoicehead") + " (TEST)");
        }
        String subsitutedHtml = format(content, substitutionMap);

        // revert escaped % signs in html
        subsitutedHtml = subsitutedHtml.replace("%%", "%");

        try {
            Files.write(Paths.get(intermediateOutputHtml), subsitutedHtml.getBytes());
        } catch (IOException e1) {
            e1.printStackTrace();
            throw new RuntimeException("Error saving invoice html to file.");
        }

        try (OutputStream os = new FileOutputStream(invoicePath)) {
            // generate the output pdf file
            File inputHTML = new File(intermediateOutputHtml);
            Document doc = null;
            doc = Jsoup.parse(inputHTML, "UTF-8");
            doc.outputSettings().syntax(Document.OutputSettings.Syntax.xml);

            PdfRendererBuilder builder = new PdfRendererBuilder();
            builder.withUri(invoicePath);
            builder.toStream(os);
            org.w3c.dom.Document d = new W3CDom().fromJsoup(doc);
            builder.withW3cDocument(d, "/");
            builder.run();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            throw new RuntimeException("Error saving invoice to file.");
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException("Error saving invoice to file.");
        }

        return invoicePath;
    }

    public void removeFailedInvoiceFromDisk(String invoiceName) {

        System.out.println("Removing failed invoice from disk (" + invoiceName + ")...");

        if (invoiceName == null) {
            return;
        }

        String GEOREPO_INVOICING_BASEPATH = null;
        if (servletContext != null) {
            String GEOREPO_SERVER_BASE_PATH = servletContext.getInitParameter("GEOREPO_SERVER_BASE_PATH");
            if (GEOREPO_SERVER_BASE_PATH == null) {
                throw new RuntimeException("GEOREPO_SERVER_BASE_PATH is null");
            }
            GEOREPO_INVOICING_BASEPATH = GEOREPO_SERVER_BASE_PATH + "/invoices";
        } else if (altInvoicingBasepath != null) {
            GEOREPO_INVOICING_BASEPATH = altInvoicingBasepath;
        } else {
            throw new RuntimeException("Missing servlet context or invoicing basepath.");
        }

        String year = String.valueOf(invoiceYear);
        String outputDir = GEOREPO_INVOICING_BASEPATH + "/" + year + "/";
        File intermediateOutputHtml = new File(outputDir + invoiceName + ".html");
        File outputPdfFile = new File(outputDir + invoiceName + ".pdf");

        try {
            if (intermediateOutputHtml.exists()) {
                intermediateOutputHtml.delete();
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("Error removing invoice file: " + intermediateOutputHtml.getAbsolutePath()
                    + ". Assuming, it does not exist.");
        }

        try {
            if (outputPdfFile.exists()) {
                outputPdfFile.delete();
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("Error removing invoice file: " + outputPdfFile.getAbsolutePath()
                    + ". Assuming, it does not exist.");
        }
    }

    /**
     * Text placeholder substitution. See
     * https://stackoverflow.com/a/27815924/3271380
     * 
     * @param format
     * @param values
     * @return
     */
    public static String format(String format, Map<String, String> values) {
        StringBuilder formatter = new StringBuilder(format);
        List<String> valueList = new ArrayList<String>();

        Matcher matcher = Pattern.compile("\\$\\{(\\w+)}").matcher(format);

        while (matcher.find()) {
            String key = matcher.group(1);

            String formatKey = String.format("${%s}", key);
            int index = formatter.indexOf(formatKey);

            if (index != -1) {
                formatter.replace(index, index + formatKey.length(), "%s");
                valueList.add(values.get(key));
            }
        }

        return String.format(formatter.toString(), valueList.toArray());
    }

    private String readFile(String path, Charset encoding)
            throws IOException {
        byte[] encoded = Files.readAllBytes(Paths.get(path));
        return new String(encoded, encoding);
    }

    /**
     * Creates the subsitution map for the invoice parameters with some pre-defined
     * values.
     * 
     */
    public static Map<String, String> createSubstitutionMap(InvoiceFormat invoiceFormat,
            String vendorname,
            String vendorstreet,
            String vendorhousenumber,
            String vendorpostcode,
            String vendorcity,
            String customername,
            String customerstreet,
            String customerhousenumber,
            String customerpostcode,
            String customercity,
            String customercountry) {

        if (vendorname == null ||
                vendorstreet == null ||
                vendorpostcode == null ||
                vendorcity == null ||
                customername == null ||
                customerstreet == null ||
                customerpostcode == null ||
                customercity == null) {
            throw new IllegalArgumentException("Invoicing parameters are not correctly filled.");
        }

        DateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy");
        if (invoiceFormat == InvoiceFormat.german) {
            dateFormat = new SimpleDateFormat("dd.MM.yyyy");
        }
        Date date = new Date();
        String invoiceDate = dateFormat.format(date);

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.MONTH, 3);
        cal.add(Calendar.DAY_OF_MONTH, -1);
        Date date3Month = cal.getTime();
        String invoiceDate3Month = dateFormat.format(date3Month);

        Map<String, String> values = new HashMap<String, String>();
        values.put("vendorname", vendorname);
        values.put("vendorstreet", vendorstreet);
        values.put("vendorhousenumber", vendorhousenumber == null ? "" : vendorhousenumber);
        values.put("vendorpostcode", vendorpostcode);
        values.put("vendorcity", vendorcity);
        values.put("customername", customername);
        values.put("customerstreet", customerstreet);
        values.put("customerhousenumber", customerhousenumber == null ? "" : customerhousenumber);
        values.put("customerpostcode", customerpostcode);
        values.put("customercity", customercity);
        values.put("customercountry", customercountry == null ? "" : customercountry);
        values.put("invoicetitle", invoiceFormat == InvoiceFormat.german ? "Rechnungs-Nr." : "Invoice nr.");
        values.put("invoicehead", invoiceFormat == InvoiceFormat.german ? "Rechnung" : "Invoice");
        values.put("invoicedate", invoiceDate);
        values.put("date3Month", invoiceDate3Month);

        return values;
    }

    /**
     * Create the product name to be used in the invoicing process
     * 
     * @return
     */
    private String retrieveFinalProductName() {
        if (productName == null) {
            throw new IllegalArgumentException("Product name not set.");
        }
        return productName + (testMode ? "-##TEST##" : "");
    }

    // getter

    public Map<String, String> getSubstitutionMap() { return substitutionMap; }

    public String getInvoiceName() { return invoiceName; }

    public String getInvoicePath() { return invoicePath; }

    public AppKey getAppCustomer() { return appCustomer; }

    public AppKey getAppProvider() { return appProvider; }

    public String getProductName() {

        // cannot tell you the invoice pattern before the product name has been set
        if (productName == null) {
            throw new IllegalArgumentException("InvoiceGenerator: set productName first.");
        }

        // once the invoice pattern has been told, we have to fix it
        // that means, some setters on this instance may not be used anymore
        this.productNameIsFixed = true;

        return retrieveFinalProductName();
    }

    // setter

    public void setDebugMode(boolean debugMode) { this.debugMode = debugMode; }

    public void setAltInvoicingBasepath(String altInvoicingBasepath) {
        this.altInvoicingBasepath = altInvoicingBasepath;
    }

    public void setSubstitutionMap(Map<String, String> substitutionMap) { this.substitutionMap = substitutionMap; }

    public void setAppCustomer(AppKey appCustomer) { this.appCustomer = appCustomer; }

    public void setAppProvider(AppKey appProvider) { this.appProvider = appProvider; }

    public void setProductName(String productName) {
        if (this.productNameIsFixed) {
            throw new IllegalArgumentException("Cannot change product name. Product name is fixed.");
        }
        this.productName = productName;
    }

    public void setTestMode(boolean testMode) {
        if (this.productNameIsFixed) {
            throw new IllegalArgumentException("Cannot change testmode. Test mode is fixed.");
        }
        this.testMode = testMode;
    }

//    public static void main(String[] args) throws IOException {
//
//        String GEOREPO_INVOICING_BASEPATH = "/home/ulrich/temp/uma/invoices";
//
//        InvoiceGenerator invoicer = new InvoiceGenerator(null);
//        invoicer.debugMode = true;
//        invoicer.altInvoicingBasepath = GEOREPO_INVOICING_BASEPATH;
//        invoicer.productName = "georepo";
//        invoicer.testMode = false;
//
//        Integer monthlyInvoiceCounter = 5;
//        InvoiceFormat invoiceFormat = InvoiceFormat.german;
//        Map<String, String> substitutionMap = createSubstitutionMap(
//                invoiceFormat,
//                "narimo systems",
//                "Bertolt-Brecht-Allee",
//                "24",
//                "01309",
//                "Dresden",
//                "Customer",
//                "cusomterstreet.", "100", "01234", "customercity",
//                String.valueOf(monthlyInvoiceCounter));
//
//        invoicer.substitutionMap = substitutionMap;
//        invoicer.generateInvoice(monthlyInvoiceCounter, invoiceFormat);
//    }
}
