package de.narimo.georepo.server.repository;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;

import org.apache.commons.lang3.RandomStringUtils;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.commons.json.JsonConverter;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.AppBaseLayersConfiguration;
import de.narimo.georepo.server.api.AppDetailsConfiguration;
import de.narimo.georepo.server.api.app.AppManifestConfiguration;
import de.narimo.georepo.server.appconfig.AppConfiguration;
import de.narimo.georepo.server.appconfig.AppKey;
import de.narimo.georepo.server.appconfig.AppLayerConfiguration;
import de.narimo.georepo.server.appconfig.AppLayersConfiguration;
import de.narimo.georepo.server.appconfig.ExtendedAppKey;
import de.narimo.georepo.server.appconfig.GeorepoConfigAuth;
import de.narimo.georepo.server.appconfig.GeorepoConfigImageSize;
import de.narimo.georepo.server.appconfig.GeorepoConfigLayerProperties;
import de.narimo.georepo.server.appconfig.GeorepoConfigSearch;
import de.narimo.georepo.server.appconfig.GeorepoConfigTag;
import de.narimo.georepo.server.tools.TableTools;

public class AppSettingsRepository {

    public static List<AppKey> getMyAppKeys(User user) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            StringBuilder s = new StringBuilder();
            s.append("SELECT appkey, createdat, activatedat FROM public.appkeys"
                    + " WHERE createdby = " + user.getId() + ";");

            PreparedStatement p = jdbcAuth.prepareStatement(s.toString());
            ResultSet rs = p.executeQuery();

            List<AppKey> appkeys = new ArrayList<>();

            while (rs.next()) {
                AppKey dbappkey = new AppKey();
                dbappkey.setAppKey(rs.getString("appkey"));
                dbappkey.setCreatedAt(rs.getString("createdat"));
                dbappkey.setActivatedAt(rs.getString("activatedat"));

                appkeys.add(dbappkey);
            }
            return appkeys;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error reading appkeys.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppKey getAppKey(String appkey) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            StringBuilder s = new StringBuilder();
            s.append(
                    "SELECT appkey, alias, createdby, createdat, activatedat, mailinghost, mailinguser, mailingpw FROM public.appkeys"
                            + " WHERE appkey = ?;");

            PreparedStatement p = jdbcAuth.prepareStatement(s.toString());
            p.setString(1, appkey);
            ResultSet rs = p.executeQuery();

            if (rs.next()) {
                AppKey dbappkey = new AppKey();
                dbappkey.setAppKey(rs.getString("appkey"));
                dbappkey.setAlias(rs.getString("alias"));
                dbappkey.setCreatedBy(rs.getInt("createdby"));
                dbappkey.setCreatedAt(rs.getString("createdat"));
                dbappkey.setActivatedAt(rs.getString("activatedat"));
                dbappkey.setMailingHost(rs.getString("mailinghost"));
                dbappkey.setMailingUser(rs.getString("mailinguser"));
                dbappkey.setMailingPw(rs.getString("mailingpw"));
                return dbappkey;
            } else {
                // we might want to handle what happens, if no appkey found
                return null;
            }
//            throw new NotFoundException();

        } catch (NotFoundException e) {
            e.printStackTrace();
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error reading appkeys.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    /**
     * Returns information about an appkey that can be publicly handed out.
     */
    public static ExtendedAppKey getPublicAppKeyInformation(String appkey, User user) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            boolean isAppOwner = isAppOwner(appkey, user);

            String sql = "SELECT appkey, createdat, activatedat, templatekey, "
                    + " organisation, address_street, address_streetnumber, address_postcode, address_city, address_country"
                    + " FROM public.appkeys WHERE appkey = ?;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            ResultSet rs = ps.executeQuery();

            String templateKey = null;
            Boolean isAppActivated = false;
            ExtendedAppKey dbappkey = new ExtendedAppKey();

            if (rs.next()) {
                dbappkey.setAppKey(rs.getString("appkey"));
                dbappkey.setCreatedAt(rs.getString("createdat"));
                dbappkey.setActivatedAt(rs.getString("activatedat"));

                templateKey = rs.getString("templatekey");
                dbappkey.setTemplateKey(templateKey);

                isAppActivated = rs.getString("activatedat") != null;

                if (isAppActivated || isAppOwner) {
                    dbappkey.setOrganisation(rs.getString("organisation"));
                    dbappkey.setAddressStreet(rs.getString("address_street"));
                    dbappkey.setAddressStreetnumber(rs.getString("address_streetnumber"));
                    dbappkey.setAddressPostcode(rs.getString("address_postcode"));
                    dbappkey.setAddressCity(rs.getString("address_city"));
                    dbappkey.setAddressCountry(rs.getString("address_country"));
                }
            } else {
                throw new NotFoundException("AppKey " + appkey + " not found.");
            }

            if (templateKey != null && (isAppActivated || isAppOwner)) {
                String sql1 = "SELECT appkey, createdat, activatedat, "
                        + " organisation, address_street, address_streetnumber, address_postcode, address_city, address_country"
                        + " FROM public.appkeys WHERE appkey = ?;";

                PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
                ps1.setString(1, templateKey);
                ResultSet rs1 = ps1.executeQuery();

                if (rs1.next()) {
                    dbappkey.setProviderOrganisation(rs1.getString("organisation"));
                    dbappkey.setProviderAddressStreet(rs1.getString("address_street"));
                    dbappkey.setProviderAddressStreetnumber(rs1.getString("address_streetnumber"));
                    dbappkey.setProviderAddressPostcode(rs1.getString("address_postcode"));
                    dbappkey.setProviderAddressCity(rs1.getString("address_city"));
                    dbappkey.setProviderAddressCountry(rs1.getString("address_country"));
                }
            }
            return dbappkey;
        } catch (NotFoundException e) {
            e.printStackTrace();
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error reading appkey.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    /**
     * Returns all appkeys that have been created from a given template app key or
     * is the template key itself. Only activated appkeys will be returned.
     * 
     * @param templateKey
     * @return
     */
    public static List<AppKey> getAppKeysByTemplateKey(String templateKey) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT distinct(k.appkey), k.templatekey, k.createdat, c.appname FROM public.appkeys k, public.appconfigs c"
                    + " WHERE k.appkey = c.appkey"
                    + " AND ((k.templatekey = ? AND k.activatedat IS NOT NULL)"
                    + " OR k.appkey = ?);";

            PreparedStatement p = jdbcAuth.prepareStatement(sql);
            p.setString(1, templateKey);
            p.setString(2, templateKey);
            System.out.println(p.toString());
            ResultSet rs = p.executeQuery();

            List<AppKey> appkeys = new ArrayList<>();

            while (rs.next()) {
                AppKey appkey = new AppKey();
                appkey.setAppKey(rs.getString("appkey"));
                appkey.setTemplateKey(rs.getString("templatekey"));
                appkey.setTitle(rs.getString("appname"));
                appkey.setCreatedAt(rs.getString("createdat"));
                appkeys.add(appkey);
            }
            return appkeys;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error reading appkeys.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static String createAppKey(User user, AppKey appkey) {
        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String templateAlias = null;

            if (appkey.getTemplateKey() != null) {
                PreparedStatement ps0 = jdbcAuth
                        .prepareStatement("SELECT appkey, alias FROM public.appkeys WHERE appkey=?");
                ps0.setString(1, appkey.getTemplateKey());

                ResultSet rs0 = ps0.executeQuery();

                if (!rs0.next()) {
                    throw new NotFoundException("Template key does not exist.");
                }
                templateAlias = rs0.getString("alias");
            }

            String sql = "INSERT INTO public.appkeys (templatekey, organisation, ownername, address_street, address_streetnumber, "
                    + "address_postcode, address_city, createdby, appkey, alias, address_country)"
                    + " VALUES(?,?,?,?,?,?,?,?,?,?,?) RETURNING appkey;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey.getTemplateKey());
            ps.setString(2, appkey.getOrganisation());
            ps.setString(3, appkey.getOwnerName());
            ps.setString(4, appkey.getAddressStreet());
            ps.setString(5, appkey.getAddressStreetnumber());
            ps.setString(6, appkey.getAddressPostcode());
            ps.setString(7, appkey.getAddressCity());
            ps.setInt(8, user.getId());
            ps.setString(9, createAppKey());

            // set the alias for a new app optionally equal to the alias of the template key
            String alias = appkey.getAlias();
            if (alias == null) {
                alias = templateAlias;
            }
            ps.setString(10, alias);
            ps.setString(11, appkey.getAddressCountry());

            ResultSet rs = ps.executeQuery();

            if (rs.next()) {
                // to reward the user for requesting a new appkey, we'll give him +1 on the
                // allowedworkspace count
                UserRepository.incrementAllowedWorkspaceCount(user);

                return rs.getString("appkey");
            }
            throw new RuntimeException("Could not insert appkey.");

        } catch (NotFoundException e) {
            e.printStackTrace();
            throw new NotFoundException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Cannot create appkey.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    private static String createAppKey() { return RandomStringUtils.randomAlphanumeric(32).toLowerCase(); }

    public static void saveAppConfiguration(AppConfiguration appConfiguration) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            String appSettingsTable = TableTools.getAppConfigurationTableName();

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "INSERT INTO public." + appSettingsTable
                    + " (appkey, appname, appnameinternal, appversion, appicon, appurl, georeposearch, "
                    + "georepoauth, appfooter, routing, geolocation, geolocationfeaturesvisible, serverurl, usecookies,"
                    + "navigationservice, featurespropertieswindow, startzoomusercentered, requireauth, registerusersonworkspace,"
                    + "namecomponentfeaturedetail,sliderautostart, ispublicapp, "
                    + "activeaddnewpoiatlocation, defaultleftdrawercomponent, leftdrawercomponent,"
                    + "rightdrawercomponent, comments, documents, links,"
                    + "sizelimitimagewidth, sizelimitimageheight, timezone, timeformat, settingsdrawer,"
                    + "defaultsettingspage, fontsize, thememode, primarycolor, secondarycolor, accentcolor, shortprecisioncoordinates,"
                    + "tagssearch, observationssearch, observationstitle, openobservations,"
                    + "aboutpagetitle, isorderapp, defaultLanguage) "
                    + "VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) "
                    + "ON CONFLICT (appkey) "
                    + "DO UPDATE SET "
                    + "appname=EXCLUDED.appname, appnameinternal=EXCLUDED.appnameinternal, appversion=EXCLUDED.appversion, appicon=EXCLUDED.appicon, appurl=EXCLUDED.appurl, georeposearch=EXCLUDED.georeposearch, "
                    + "georepoauth=EXCLUDED.georepoauth, appfooter=EXCLUDED.appfooter, routing=EXCLUDED.routing, geolocation=EXCLUDED.geolocation, geolocationfeaturesvisible=EXCLUDED.geolocationfeaturesvisible, serverurl=EXCLUDED.serverurl, usecookies=EXCLUDED.usecookies,"
                    + "navigationservice=EXCLUDED.navigationservice, featurespropertieswindow=EXCLUDED.featurespropertieswindow, startzoomusercentered=EXCLUDED.startzoomusercentered, requireauth=EXCLUDED.requireauth, registerusersonworkspace=EXCLUDED.registerusersonworkspace,"
                    + "namecomponentfeaturedetail=EXCLUDED.namecomponentfeaturedetail,sliderautostart=EXCLUDED.sliderautostart, ispublicapp=EXCLUDED.ispublicapp, "
                    + "activeaddnewpoiatlocation=EXCLUDED.activeaddnewpoiatlocation, defaultleftdrawercomponent=EXCLUDED.defaultleftdrawercomponent, leftdrawercomponent=EXCLUDED.leftdrawercomponent,"
                    + "rightdrawercomponent=EXCLUDED.rightdrawercomponent, comments=EXCLUDED.comments, documents=EXCLUDED.documents, links=EXCLUDED.links,"
                    + "sizelimitimagewidth=EXCLUDED.sizelimitimagewidth, sizelimitimageheight=EXCLUDED.sizelimitimageheight, timezone=EXCLUDED.timezone, timeformat=EXCLUDED.timeformat, settingsdrawer=EXCLUDED.settingsdrawer,"
                    + "defaultsettingspage=EXCLUDED.defaultsettingspage, fontsize=EXCLUDED.fontsize, thememode=EXCLUDED.thememode, primarycolor=EXCLUDED.primarycolor, secondarycolor=EXCLUDED.secondarycolor, accentcolor=EXCLUDED.accentcolor, shortprecisioncoordinates=EXCLUDED.shortprecisioncoordinates,"
                    + "tagssearch=EXCLUDED.tagssearch, observationssearch=EXCLUDED.observationssearch, observationstitle=EXCLUDED.observationstitle, openobservations=EXCLUDED.openobservations,"
                    + "aboutpagetitle=EXCLUDED.aboutpagetitle, isorderapp=EXCLUDED.isorderapp, defaultLanguage=EXCLUDED.defaultLanguage"
                    + ";";

            int i = 1;
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(i++, appConfiguration.getAppKey());
            ps.setString(i++, appConfiguration.getAppName());
            ps.setString(i++, appConfiguration.getAppNameInternal());
            ps.setString(i++, appConfiguration.getAppVersion());
            ps.setString(i++, appConfiguration.getAppIcon());
            ps.setString(i++, appConfiguration.getAppUrl());
            ps.setString(i++, JsonConverter.toJson(appConfiguration.getSearch()));
            ps.setString(i++, JsonConverter.toJson(appConfiguration.getAuth()));
            ps.setBoolean(i++, appConfiguration.getAppFooter() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getRouting() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getGeolocation() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getGeolocationFeaturesVisible() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getServerUrl());
            ps.setBoolean(i++, appConfiguration.getUseCookies() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getNavigationService());
            ps.setString(i++, appConfiguration.getFeaturesPropertiesWindow());
            ps.setBoolean(i++, appConfiguration.getStartZoomUserCentered() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getRequireAuth() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getRegisterUsersOnWorkspace() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getNameComponentFeatureDetail());
            ps.setBoolean(i++, appConfiguration.getSliderAutostart() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getIsPublicApp() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getActiveAddNewPoiAtLocation() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getDefaultLeftDrawerComponent());
            ps.setString(i++, JsonConverter.toJson(appConfiguration.getLeftDrawerComponent()));
            ps.setString(i++, JsonConverter.toJson(appConfiguration.getRightDrawerComponent()));
            ps.setBoolean(i++, appConfiguration.getComments() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getDocuments() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getLinks() == Boolean.TRUE);
            ps.setInt(i++, appConfiguration.getSizeLimitImage().getWidth());
            ps.setInt(i++, appConfiguration.getSizeLimitImage().getHeight());
            ps.setString(i++, appConfiguration.getTimeZone());
            ps.setString(i++, appConfiguration.getTimeFormat());
            ps.setBoolean(i++, appConfiguration.getSettingsDrawer() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getDefaultSettingsPage());
            ps.setInt(i++, appConfiguration.getFontSize());
            ps.setString(i++, appConfiguration.getThemeMode());
            ps.setString(i++, appConfiguration.getPrimaryColor());
            ps.setString(i++, appConfiguration.getSecondaryColor());
            ps.setString(i++, appConfiguration.getAccentColor());
            ps.setInt(i++, appConfiguration.getShortPrecisionCoordinates());
            ps.setBoolean(i++, appConfiguration.getTagsSearch() == Boolean.TRUE);
            ps.setBoolean(i++, appConfiguration.getObservationsSearch() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getObservationsTitle());
            ps.setBoolean(i++, appConfiguration.getOpenObservations() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getAboutPageTitle());
            ps.setBoolean(i++, appConfiguration.getIsOrderApp() == Boolean.TRUE);
            ps.setString(i++, appConfiguration.getDefaultLanguage());

            System.out.println(ps.toString());
            ps.execute();

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not insert app config.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void saveAppLayersConfiguration(AppLayersConfiguration appLayersConfiguration) {

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            String appLayerTable = TableTools.getAppLayerTableName();

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            List<String> layerIdsToSave = new ArrayList<>();

            // TODO: enhance insert all layer configs in one statement
            for (AppLayerConfiguration appLayerConfiguration : appLayersConfiguration.getAppLayerConfigurations()) {
                layerIdsToSave.add(appLayerConfiguration.getId());

                String sql = "INSERT INTO public." + appLayerTable
                        + " (appkey, id, name, icon, type, workspace, georeponame, categories, tags, visible, legend, edit, "
                        + "translate, clearable, selectable, navigation, class, declutter, searchable, primaryLayer, nameProperty, "
                        + "categoryProperty, areaProperty, strokeColor, strokeWidth, fillColor, properties, style, fromLocal) "
                        + "VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?) "
                        + "ON CONFLICT (appkey,id) "
                        + "DO UPDATE SET "
                        + "name=EXCLUDED.name, icon=EXCLUDED.icon, type=EXCLUDED.type, workspace=EXCLUDED.workspace, "
                        + "georeponame=EXCLUDED.georeponame, categories=EXCLUDED.categories, tags=EXCLUDED.tags, visible=EXCLUDED.visible, "
                        + "legend=EXCLUDED.legend, edit=EXCLUDED.edit, translate=EXCLUDED.translate, clearable=EXCLUDED.clearable, "
                        + "selectable=EXCLUDED.selectable, navigation=EXCLUDED.navigation, class=EXCLUDED.class, declutter=EXCLUDED.declutter, "
                        + "searchable=EXCLUDED.searchable, primaryLayer=EXCLUDED.primaryLayer, nameProperty=EXCLUDED.nameProperty, "
                        + "categoryProperty=EXCLUDED.categoryProperty, areaProperty=EXCLUDED.areaProperty, "
                        + "strokeColor=EXCLUDED.strokeColor, strokeWidth=EXCLUDED.strokeWidth, fillColor=EXCLUDED.fillColor, "
                        + "properties=EXCLUDED.properties, style=EXCLUDED.style, fromLocal=EXCLUDED.fromLocal;";

                int i = 1;
                PreparedStatement ps = jdbcAuth.prepareStatement(sql);
                ps.setString(i++, appLayersConfiguration.getAppKey());
                ps.setString(i++, appLayerConfiguration.getId());
                ps.setString(i++, appLayerConfiguration.getName());
                ps.setString(i++, appLayerConfiguration.getIcon());
                ps.setString(i++, appLayerConfiguration.getType());
                ps.setString(i++, appLayerConfiguration.getWorkspace());
                ps.setString(i++, appLayerConfiguration.getGeoreponame());
                ps.setString(i++, JsonConverter.toJson(appLayerConfiguration.getCategories()));
                ps.setString(i++, JsonConverter.toJson(appLayerConfiguration.getTags()));
                ps.setBoolean(i++, appLayerConfiguration.getVisible() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getLegend() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getEdit() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getTranslate() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getClearable() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getSelectable() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getNavigation() == Boolean.TRUE);
                ps.setString(i++, appLayerConfiguration.getClassType());
                ps.setBoolean(i++, appLayerConfiguration.getDeclutter() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getSearchable() == Boolean.TRUE);
                ps.setBoolean(i++, appLayerConfiguration.getPrimaryLayer() == Boolean.TRUE);
                ps.setString(i++, appLayerConfiguration.getNameProperty());
                ps.setString(i++, appLayerConfiguration.getCategoryProperty());
                ps.setString(i++, appLayerConfiguration.getAreaProperty());
                ps.setString(i++, appLayerConfiguration.getStrokeColor());
                ps.setString(i++, appLayerConfiguration.getStrokeWidth());
                ps.setString(i++, appLayerConfiguration.getFillColor());
                ps.setString(i++, JsonConverter.toJson(appLayerConfiguration.getProperties()));
                ps.setString(i++, appLayerConfiguration.getStyle());
                ps.setBoolean(i++, appLayerConfiguration.getFromLocal() == Boolean.TRUE);

                System.out.println(ps.toString());
                ps.execute();
            }

            // Delete layer ids for the given appkey, that does not exist any more in the
            // configuration
            String sql1 = "SELECT * FROM public." + appLayerTable + " WHERE appkey=?";
            PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
            ps1.setString(1, appLayersConfiguration.getAppKey());
            ResultSet rs = ps1.executeQuery();
            while (rs.next()) {
                String layerId = rs.getString("id");
                if (!layerIdsToSave.contains(layerId)) {
                    String sql2 = "DELETE FROM public." + appLayerTable + " WHERE appkey=? and id=?";
                    PreparedStatement ps2 = jdbcAuth.prepareStatement(sql2);
                    ps2.setString(1, appLayersConfiguration.getAppKey());
                    ps2.setString(2, layerId);
                    ps2.executeUpdate();
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not insert app layers config.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void saveAppBaseLayersConfiguration(AppBaseLayersConfiguration appBaseLayersConfiguration) {
        JDBCConnectionJNDI jdbcAuth = null;
        try {
            String appBaseLayerTable = TableTools.getAppBaseLayerTableName();

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "INSERT INTO public." + appBaseLayerTable
                    + " (appkey, baselayers) "
                    + "VALUES(?,?) "
                    + "ON CONFLICT (appkey) "
                    + "DO UPDATE SET baselayers=EXCLUDED.baselayers;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appBaseLayersConfiguration.getAppKey());
            ps.setString(2, JsonConverter.toJson(appBaseLayersConfiguration.getAppBaseLayerConfigurations()));

            System.out.println(ps.toString());
            ps.execute();

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not insert base layers config.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void saveAppDetailsConfiguration(AppDetailsConfiguration appDetailsConfiguration) {
        JDBCConnectionJNDI jdbcAuth = null;
        try {
            String appDetailsTable = TableTools.getAppDetailsTableName();

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "INSERT INTO public." + appDetailsTable
                    + " (appkey, appdetails) "
                    + "VALUES(?,?) "
                    + "ON CONFLICT (appkey) "
                    + "DO UPDATE SET appdetails=EXCLUDED.appdetails;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appDetailsConfiguration.getAppKey());
            ps.setString(2, JsonConverter.toJson(appDetailsConfiguration.getAppDetails()));

            System.out.println(ps.toString());
            ps.execute();

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not insert app details config.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static boolean isAppOwner(String appkey, User user) {
        if (appkey == null) {
            return false;
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT appkey FROM public.appkeys WHERE appkey = ? and createdby = " + user.getId() + ";";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (!rs.next()) {
                return false;
            }
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Error getting app ownership info.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void createAppConfigTable() throws SQLException {
        JDBCConnectionJNDI jdbcAuth = null;

        String appConfigTable = TableTools.getAppConfigurationTableName();

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcAuth, "public", appConfigTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + appConfigTable + "("
                    + "id serial PRIMARY KEY, appkey text NOT NULL UNIQUE, appname text, appnameinternal text, appversion text, appicon text, appurl text, georeposearch text, "
                    + "georepoauth text, appfooter boolean default true, routing boolean default false, geolocation boolean default false, geolocationfeaturesvisible  boolean default false, serverurl text, usecookies boolean default false,"
                    + "navigationservice text, featurespropertieswindow text, startzoomusercentered boolean, requireauth boolean, registerusersonworkspace boolean,"
                    + "namecomponentfeaturedetail text,sliderautostart boolean, ispublicapp boolean default false, "
                    + "activeaddnewpoiatlocation boolean default false, defaultleftdrawercomponent text, leftdrawercomponent text,"
                    + "rightdrawercomponent text, comments boolean default false, documents boolean default false, links boolean default false,"
                    + "sizelimitimagewidth integer, sizelimitimageheight integer, timezone text, timeformat text, settingsdrawer boolean default false,"
                    + "defaultsettingspage text, fontsize integer, thememode text, primarycolor text, secondarycolor text, accentcolor text, shortprecisioncoordinates integer,"
                    + "tagssearch boolean default false, observationssearch boolean default false, observationstitle text, openobservations boolean default false,"
                    + "aboutpagetitle text, isorderapp boolean default false, defaultlanguage text"
                    + ");";

            System.out.println(sql);
            jdbcAuth.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void createAppLayerConfigTable() throws SQLException {
        JDBCConnectionJNDI jdbcAuth = null;

        String appLayerTable = TableTools.getAppLayerTableName();

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcAuth, "public", appLayerTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + appLayerTable + "("
                    + "layerid serial PRIMARY KEY, appkey text NOT NULL, id text, name text, icon text, type text, workspace text, georeponame text, categories text, tags text, "
                    + "visible boolean default false, legend boolean default false, edit boolean default false, translate boolean default false, "
                    + "clearable boolean default false, selectable boolean default false, navigation boolean default false, class text, declutter boolean default false, "
                    + "searchable boolean default false, primaryLayer boolean default false, nameProperty text, categoryProperty text, areaProperty text, descriptionProperty text, "
                    + "strokeColor text, strokeWidth text, fillColor text, properties text, style text, fromlocal boolean"
                    + ");"
                    + "CREATE UNIQUE INDEX IF NOT EXISTS uniq_idx_" + appLayerTable
                    + "_appkey_id  ON " + appLayerTable + "(appkey, id);";

            System.out.println(sql);
            jdbcAuth.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void createAppBaseLayerConfigTable() throws SQLException {
        String appBaseLayerTable = TableTools.getAppBaseLayerTableName();

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcAuth, "public", appBaseLayerTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + appBaseLayerTable + "("
                    + "id serial PRIMARY KEY, appkey text UNIQUE NOT NULL, baselayers text);";
            System.out.println(sql);
            jdbcAuth.execute(sql);
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void createAppDetailsConfigTable() throws SQLException {
        String appDetailsTable = TableTools.getAppDetailsTableName();

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcAuth, "public", appDetailsTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + appDetailsTable + "("
                    + "id serial PRIMARY KEY, appkey text UNIQUE NOT NULL, appdetails text);";
            System.out.println(sql);
            jdbcAuth.execute(sql);
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void createAppManifestTable() throws SQLException {
        String appManifestTable = TableTools.getAppManifestTableName();

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (ParentDatasetRepository.tableExists(jdbcAuth, "public", appManifestTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + appManifestTable + "("
                    + "id serial PRIMARY KEY, appkey text UNIQUE NOT NULL, manifest text);";
            System.out.println(sql);
            jdbcAuth.execute(sql);
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppConfiguration getAppConfiguration(String appkey) throws SQLException {
        String appSettingsTable = TableTools.getAppConfigurationTableName();
        JDBCConnectionJNDI jdbcAuth = null;

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT * FROM public." + appSettingsTable + " WHERE appkey = ?;";
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            System.out.println(ps.toString());
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (rs.next()) {
                AppConfiguration appConfig = new AppConfiguration();
                appConfig.setAppName(rs.getString("appname"));
                appConfig.setAppNameInternal(rs.getString("appnameinternal"));
                appConfig.setAppVersion(rs.getString("appversion"));
                appConfig.setAppIcon(rs.getString("appicon"));
                appConfig.setAppUrl(rs.getString("appurl"));
                appConfig.setSearch(JsonConverter.fromJson(rs.getString("georeposearch"), GeorepoConfigSearch.class));
                appConfig.setAuth(JsonConverter.fromJson(rs.getString("georepoauth"), GeorepoConfigAuth.class));
                appConfig.setAppFooter(rs.getBoolean("appfooter"));
                appConfig.setRouting(rs.getBoolean("routing"));
                appConfig.setGeolocation(rs.getBoolean("geolocation"));
                appConfig.setGeolocationFeaturesVisible(rs.getBoolean("geolocationfeaturesvisible"));
                appConfig.setServerUrl(rs.getString("serverurl"));
                appConfig.setUseCookies(rs.getBoolean("usecookies"));
                appConfig.setNavigationService(rs.getString("navigationservice"));
                appConfig.setFeaturesPropertiesWindow(rs.getString("featurespropertieswindow"));
                appConfig.setStartZoomUserCentered(rs.getBoolean("startzoomusercentered"));
                appConfig.setRequireAuth(rs.getBoolean("RequireAuth"));
                appConfig.setRegisterUsersOnWorkspace(rs.getBoolean("registerusersonworkspace"));
                appConfig.setNameComponentFeatureDetail(rs.getString("namecomponentfeaturedetail"));
                appConfig.setSliderAutostart(rs.getBoolean("sliderautostart"));
                appConfig.setIsPublicApp(rs.getBoolean("ispublicapp"));
                appConfig.setActiveAddNewPoiAtLocation(rs.getBoolean("activeaddnewpoiatlocation"));
                appConfig.setDefaultLeftDrawerComponent(rs.getString("defaultleftdrawercomponent"));
                appConfig.setLeftDrawerComponent(
                        JsonConverter.fromJsonToList(rs.getString("leftdrawercomponent"), String.class));
                appConfig.setRightDrawerComponent(
                        JsonConverter.fromJsonToList(rs.getString("rightdrawercomponent"), String.class));
                appConfig.setComments(rs.getBoolean("comments"));
                appConfig.setDocuments(rs.getBoolean("documents"));
                appConfig.setLinks(rs.getBoolean("links"));
                GeorepoConfigImageSize size = new GeorepoConfigImageSize();
                size.setHeight(rs.getInt("sizelimitimageheight"));
                size.setWidth(rs.getInt("sizelimitimagewidth"));
                appConfig.setSizeLimitImage(size);
                appConfig.setTimeZone(rs.getString("timezone"));
                appConfig.setTimeFormat(rs.getString("timeformat"));
                appConfig.setSettingsDrawer(rs.getBoolean("settingsdrawer"));
                appConfig.setDefaultSettingsPage(rs.getString("defaultsettingspage"));
                appConfig.setFontSize(rs.getInt("fontsize"));
                appConfig.setThemeMode(rs.getString("thememode"));
                appConfig.setPrimaryColor(rs.getString("primarycolor"));
                appConfig.setSecondaryColor(rs.getString("secondarycolor"));
                appConfig.setAccentColor(rs.getString("accentcolor"));
                appConfig.setShortPrecisionCoordinates(rs.getInt("shortprecisioncoordinates"));
                appConfig.setTagsSearch(rs.getBoolean("tagssearch"));
                appConfig.setObservationsSearch(rs.getBoolean("observationssearch"));
                appConfig.setObservationsTitle(rs.getString("observationstitle"));
                appConfig.setOpenObservations(rs.getBoolean("openobservations"));
                appConfig.setAboutPageTitle(rs.getString("aboutpagetitle"));
                appConfig.setIsOrderApp(rs.getBoolean("isorderapp"));
                appConfig.setDefaultLanguage(rs.getString("defaultLanguage"));

                return appConfig;
            } else {
                throw new NotFoundException("No app configuration available.");
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppLayersConfiguration getAppLayersConfiguration(String appkey) throws SQLException {
        String appLayerTable = TableTools.getAppLayerTableName();
        JDBCConnectionJNDI jdbcAuth = null;

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT * FROM public." + appLayerTable + " WHERE appkey = ?;";
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            System.out.println(ps.toString());
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            AppLayersConfiguration appLayersConfiguration = new AppLayersConfiguration();
            appLayersConfiguration.setAppKey(appkey);
            appLayersConfiguration.setAppLayerConfigurations(new ArrayList<>());

            while (rs.next()) {
                AppLayerConfiguration layerConfig = new AppLayerConfiguration();

                layerConfig.setId(rs.getString("id"));
                layerConfig.setName(rs.getString("name"));
                layerConfig.setIcon(rs.getString("icon"));
                layerConfig.setType(rs.getString("type"));
                layerConfig.setWorkspace(rs.getString("workspace"));
                layerConfig.setGeoreponame(rs.getString("georeponame"));
                layerConfig.setCategories(JsonConverter.fromJsonToList(rs.getString("categories"), String.class));
                layerConfig.setTags(JsonConverter.fromJsonToList(rs.getString("tags"), GeorepoConfigTag.class));
                layerConfig.setVisible(rs.getBoolean("visible"));
                layerConfig.setLegend(rs.getBoolean("legend"));
                layerConfig.setEdit(rs.getBoolean("edit"));
                layerConfig.setTranslate(rs.getBoolean("translate"));
                layerConfig.setClearable(rs.getBoolean("clearable"));
                layerConfig.setSelectable(rs.getBoolean("selectable"));
                layerConfig.setNavigation(rs.getBoolean("navigation"));
                layerConfig.setClassType(rs.getString("class"));
                layerConfig.setDeclutter(rs.getBoolean("declutter"));
                layerConfig.setSearchable(rs.getBoolean("searchable"));
                layerConfig.setPrimaryLayer(rs.getBoolean("primarylayer"));
                layerConfig.setNameProperty(rs.getString("nameproperty"));
                layerConfig.setCategoryProperty(rs.getString("categoryproperty"));
                layerConfig.setAreaProperty(rs.getString("areaproperty"));
                layerConfig.setDescriptionProperty(rs.getString("descriptionproperty"));
                layerConfig.setStrokeColor(rs.getString("strokecolor"));
                layerConfig.setStrokeWidth(rs.getString("strokewidth"));
                layerConfig.setFillColor(rs.getString("fillcolor"));
                layerConfig
                        .setProperties(JsonConverter.fromJsonToList(rs.getString("properties"),
                                GeorepoConfigLayerProperties.class));
                layerConfig.setStyle(rs.getString("style"));
                layerConfig.setFromLocal(rs.getBoolean("fromlocal"));

                appLayersConfiguration.getAppLayerConfigurations().add(layerConfig);
            }
            return appLayersConfiguration;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppBaseLayersConfiguration getAppBaseLayersConfiguration(String appkey) throws SQLException {
        String appBaseLayerTable = TableTools.getAppBaseLayerTableName();
        JDBCConnectionJNDI jdbcAuth = null;

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT * FROM public." + appBaseLayerTable + " WHERE appkey = ?;";
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            System.out.println(ps.toString());
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (rs.next()) {
                AppBaseLayersConfiguration appBaseLayersConfiguration = new AppBaseLayersConfiguration();
                appBaseLayersConfiguration.setAppKey(appkey);
                appBaseLayersConfiguration.setAppBaseLayerConfigurations(
                        JsonConverter.fromJsonToList(rs.getString("baselayers"), Object.class));

                return appBaseLayersConfiguration;
            } else {
                throw new NotFoundException("No base layers found for appkey " + appkey + ".");
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppDetailsConfiguration getAppDetailsConfiguration(String appkey)
            throws NotFoundException, SQLException {
        String appDetailsTable = TableTools.getAppDetailsTableName();
        JDBCConnectionJNDI jdbcAuth = null;

        AppDetailsConfiguration appDetailsConfiguration = new AppDetailsConfiguration();
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT * FROM public." + appDetailsTable + " WHERE appkey = ?;";
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            System.out.println(ps.toString());
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (rs.next()) {
                appDetailsConfiguration.setAppKey(appkey);
                appDetailsConfiguration.setAppDetails(
                        JsonConverter.fromJsonToList(rs.getString("appdetails"), Object.class));

                return appDetailsConfiguration;
            } else {
                throw new NotFoundException("No app details found for appkey " + appkey + ".");
            }
        } catch (Exception e) {
            e.printStackTrace();
            // return empty app details
//            appDetailsConfiguration.setAppDetails(new ArrayList<>());
//            return appDetailsConfiguration;
            throw new NotFoundException("No app details found for appkey " + appkey + ".");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static boolean isAppkeyActivated(String appkey, User user) {
        if (appkey == null || user == null) {
            throw new RuntimeException("No appkey or user provided.");
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            // check if the appkey is already activated
            String sql = "SELECT activatedat FROM public.appkeys a, public.users u WHERE a.createdby = u.id AND appkey = ? and createdby = "
                    + user.getId() + " AND activatedat IS NOT NULL AND u.enabled;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (rs.next()) {
                return true;
                // throw new IllegalStateException("The provided appkey is already activated.");
            }

            return false;
        } catch (SQLException e) {
            throw new RuntimeException("Could not determine appkey status.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static Map<String, AppKey> getAppkeyInvoiceDetails(String appkey, User user) {
        if (appkey == null || user == null) {
            throw new RuntimeException("No appkey or user provided.");
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            // get customer details from the appkey
            String sql0 = "SELECT a.activatedat, "
                    + "a.ownername, a.address_street, a.address_postcode, a.address_city, a.address_streetnumber, a.address_country, "
                    + "a.organisation, a.templatekey FROM public.appkeys a, public.users u "
                    + "WHERE a.createdby = u.id AND a.appkey = ? and a.createdby = " + user.getId()
                    + " AND a.activatedat IS NULL AND u.enabled;";

            PreparedStatement ps0 = jdbcAuth.prepareStatement(sql0);
            ps0.setString(1, appkey);
            ResultSet rs0 = jdbcAuth.executePreparedQuery(ps0);

            if (!rs0.next()) {
                throw new RuntimeException("No app found for appkey " + appkey + ".");
            }

            AppKey appCustomer = new AppKey();
            appCustomer.setOwnerName(rs0.getString("ownername"));
            appCustomer.setAddressStreet(rs0.getString("address_street"));
            appCustomer.setAddressStreetnumber(rs0.getString("address_streetnumber"));
            appCustomer.setAddressPostcode(rs0.getString("address_postcode"));
            appCustomer.setAddressCity(rs0.getString("address_city"));
            appCustomer.setAddressCountry(rs0.getString("address_country"));
            appCustomer.setOrganisation(rs0.getString("organisation"));

            String templateKey = rs0.getString("templatekey");
            if (templateKey == null || templateKey.trim().equals("")) {
                throw new RuntimeException("Invalid template key " + templateKey + ".");
            }

            // get provider details from the template appkey
            String sql1 = "SELECT a.activatedat, a.alias, "
                    + "a.ownername, a.address_street, a.address_postcode, a.address_city, a.address_streetnumber, a.address_country, "
                    + "a.organisation FROM public.appkeys a, public.users u "
                    + "WHERE a.createdby = u.id AND a.appkey = ? AND a.activatedat IS NOT NULL AND u.enabled;";

            PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
            ps1.setString(1, templateKey);
            ResultSet rs1 = jdbcAuth.executePreparedQuery(ps1);

            if (!rs1.next()) {
                throw new RuntimeException(
                        "The template key " + templateKey + " does not exist. Cannot determine provider app!");
            }

            AppKey appProvider = new AppKey();
            appProvider.setOwnerName(rs1.getString("ownername"));
            appProvider.setAddressStreet(rs1.getString("address_street"));
            appProvider.setAddressStreetnumber(rs1.getString("address_streetnumber"));
            appProvider.setAddressPostcode(rs1.getString("address_postcode"));
            appProvider.setAddressCity(rs1.getString("address_city"));
            appProvider.setAddressCountry(rs1.getString("address_country"));
            appProvider.setOrganisation(rs1.getString("organisation"));
            appProvider.setAlias(rs1.getString("alias"));

            Map<String, AppKey> invoiceAppkeys = new HashMap<>();
            invoiceAppkeys.put("appcustomer", appCustomer);
            invoiceAppkeys.put("appprovider", appProvider);

            return invoiceAppkeys;
        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException("Could not determine appkey invoice details.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    /**
     * Returns the userid of the user that is the app provider, i.e. the owner of
     * the templateKey app for a given appKey.
     * 
     * @param appkey
     * @param user
     * @return
     */
    public static int getAppProvider(String appKey, User user) {
        if (appKey == null || user == null) {
            throw new RuntimeException("No appkey or user provided.");
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            // get the appkey to determine template key
            String sql = "SELECT a.createdby, a.templatekey FROM public.appkeys a, public.users u "
                    + "WHERE a.createdby = u.id AND a.appkey = ? AND a.createdby = " + user.getId()
                    + " AND a.activatedat IS NULL AND u.enabled;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appKey);
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (!rs.next() || rs.getString("templatekey") == null) {
                throw new RuntimeException(
                        "Cannot determine template key for the app key " + appKey + ".");
            }
            String templateKey = rs.getString("templatekey");

            // get provider details from the template appkey
            String sql1 = "SELECT a.createdby FROM public.appkeys a, public.users u "
                    + "WHERE a.createdby = u.id AND a.appkey = ? AND a.activatedat IS NOT NULL AND u.enabled;";

            PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
            ps1.setString(1, templateKey);
            ResultSet rs1 = jdbcAuth.executePreparedQuery(ps1);

            if (!rs1.next()) {
                throw new RuntimeException(
                        "The template key " + templateKey + " does not exist. Cannot determine provider app.");
            }

            int appCreatorId = rs1.getInt("createdby");
            return appCreatorId;
        } catch (SQLException e) {
            throw new RuntimeException("Could not determine appkey invoice details.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    /**
     * Activates an appkey. If the appkey is already activated, throws an
     * {@link IllegalStateException}
     * 
     * @param appkey
     * @param user
     * @throws ForbiddenException
     * @throws IllegalStateException
     */
    public static void activateAppkey(String appkey, User user) throws ForbiddenException, IllegalStateException {
        if (appkey == null || user == null) {
            throw new ForbiddenException();
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            if (AppSettingsRepository.isAppkeyActivated(appkey, user)) {
                throw new IllegalStateException("The provided appkey is already activated.");
            }

            String sql1 = "UPDATE public.appkeys SET activatedat = timezone('utc'::text, now()) "
                    + "WHERE appkey = ? and createdby = " + user.getId() + ";";

            PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
            ps1.setString(1, appkey);
            jdbcAuth.executePreparedUpdate(ps1);

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException("Error activating app.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    /**
     * Deactivates an appkey. Internal usage only, do not deactivate an app by user
     * request since this needs handling of invoices, etc.!
     * 
     * @param appkey
     * @param user
     * @throws ForbiddenException
     * @throws IllegalStateException
     */
    public static void deactivateAppkey(String appkey, User user) throws ForbiddenException, IllegalStateException {
        if (appkey == null || user == null) {
            throw new ForbiddenException();
        }

        JDBCConnectionJNDI jdbcAuth = null;
        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql1 = "UPDATE public.appkeys SET activatedat = null "
                    + "WHERE appkey = ? and createdby = " + user.getId() + ";";

            PreparedStatement ps1 = jdbcAuth.prepareStatement(sql1);
            ps1.setString(1, appkey);
            jdbcAuth.executePreparedUpdate(ps1);

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException("Error deactivating app.");
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static void saveAppManifest(AppManifestConfiguration appManifestConfiguration) {
        JDBCConnectionJNDI jdbcAuth = null;
        try {
            String appManifestTable = TableTools.getAppManifestTableName();

            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "INSERT INTO public." + appManifestTable
                    + " (appkey, manifest) "
                    + "VALUES(?,?) "
                    + "ON CONFLICT (appkey) "
                    + "DO UPDATE SET manifest=EXCLUDED.manifest;";

            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appManifestConfiguration.getAppKey());
            ps.setString(2, JsonConverter.toJson(appManifestConfiguration.getAppManifest()));

            System.out.println(ps.toString());
            ps.execute();

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("Could not insert or update manifest.", e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }

    public static AppManifestConfiguration getAppManifestConfiguration(String appkey)
            throws SQLException, NotFoundException {
        String appManifestTable = TableTools.getAppManifestTableName();
        JDBCConnectionJNDI jdbcAuth = null;

        try {
            jdbcAuth = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");

            String sql = "SELECT * FROM public." + appManifestTable + " WHERE appkey = ?;";
            PreparedStatement ps = jdbcAuth.prepareStatement(sql);
            ps.setString(1, appkey);
            System.out.println(ps.toString());
            ResultSet rs = jdbcAuth.executePreparedQuery(ps);

            if (rs.next()) {
                AppManifestConfiguration appManifestConfiguration = new AppManifestConfiguration();
                appManifestConfiguration.setAppKey(appkey);
                appManifestConfiguration.setAppManifest(rs.getString("manifest"));

                return appManifestConfiguration;
            } else {
                throw new NotFoundException("No manifest configuration found for appkey " + appkey + ".");
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcAuth != null) {
                jdbcAuth.closeAll();
                jdbcAuth = null;
            }
        }
    }
}
