package de.narimo.georepo.server.repository;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.api.comments.CommentOrLinkType;
import de.narimo.georepo.server.api.comments.Weblink;
import de.narimo.georepo.server.api.workspaces.Workspace;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class CommentOrWeblinkRepository {

    /**
     * Checks, whether a comment table exists for a given layer id.
     * 
     * @param layerId
     * @return
     * @throws SQLException
     */
    public static boolean checkCommentsTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String cmtTable = TableTools.getCommentTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", cmtTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static boolean checkCommentsDiffTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String cmtDiffTable = TableTools.getCommentDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", cmtDiffTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates a table that stores comments related to a POI of the layer that this
     * table is related to.
     * 
     * @param layerId
     * @throws SQLException
     */
    public static void createCommentsTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String cmtDataTable = TableTools.getCommentTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", cmtDataTable)) {
                return;
            }
            System.out.println("Comment table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + cmtDataTable + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "gfid integer NOT NULL,"
                    + "comment text,"
                    + "description text,"
                    + "type text NOT NULL,"
                    + "insertedby integer NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates a comment diff table from the structure of an comment table. And adds
     * a grpstatus column.
     *
     * @param dataTable
     */
    public static void createCommentDiffTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String commentDataTable = TableTools.getCommentTableName(layerId);
        String commentDiffTable = TableTools.getCommentDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", commentDiffTable)) {
                return;
            }
            System.out.println("Comment diff table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + commentDiffTable + ""
                    + " (LIKE " + commentDataTable + " INCLUDING DEFAULTS);"
                    + " ALTER TABLE " + commentDiffTable
                    + " DROP COLUMN id,"
                    + " ADD COLUMN id SERIAL PRIMARY KEY NOT NULL,"
                    + " ADD COLUMN " + GeorepoConstants.GRPSTATUSCOLUMN + " character varying(1) NOT NULL;";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Inserts a new comment or weblink and returns its id.
     * 
     * @param layerId
     * @param userId
     * @param gfid
     * @return
     * @throws SQLException
     */
    public static int insertUnreviewedCommentOrWeblink(int layerId, int userId, int gfid, String commentOrWeblink,
            CommentOrLinkType commentOrLinkType, String description)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String commentDiffTable = TableTools.getCommentDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "INSERT INTO " + commentDiffTable
                    + " (gfid, comment, insertedby, grpstatus, type, description) VALUES (?, ?, ?, ?, ?, ?) RETURNING id;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, commentOrWeblink);
            ps.setInt(3, userId);
            ps.setString(4, "n");
            ps.setString(5, commentOrLinkType.toString());
            ps.setString(6, description);
            ResultSet rs = ps.executeQuery();

            if (rs.next()) {
                return rs.getInt("id");
            }
            throw new RuntimeException("Could not insert comment or link.");

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static List<Weblink> getFeatureCommentsOrWeblinks(User requestingUser, String workspace, int layerId,
            int gfid,
            CommentOrLinkType commentOrLinkType)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String commentTable = TableTools.getCommentTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT id, comment, description, insertedat, insertedby FROM " + commentTable
                    + " WHERE gfid = ? AND type = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, commentOrLinkType.toString());
            ResultSet rs = ps.executeQuery();

            List<Weblink> comments = new ArrayList<>();
            List<Integer> creators = new ArrayList<>();
            while (rs.next()) {
                int id = rs.getInt("id");
                String cmt = rs.getString("comment");
                String insertedat = rs.getString("insertedat");
                String description = rs.getString("description");
                int insertedby = rs.getInt("insertedby");
                Weblink comment = new Weblink();
                comment.setId(id);
                // comment.setComment(cmt);
                comment.setLink(cmt);
                comment.setDescription(description);
                comment.setCreatedat(insertedat);
                // this is the user id but needs to be transformed to a user name
                comment.setCreator(String.valueOf(insertedby));

                creators.add(insertedby);

                comments.add(comment);
            }

            addCreatorInfo(comments, creators,
                    AdminTools.isWorkspaceAdmin(requestingUser.getId(), new Workspace(workspace)));

            return comments;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static List<Weblink> getUnreviewedCommentsOrWeblinks(User requestingUser, String workspace, int layerId,
            CommentOrLinkType commentOrLinkType)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String commentDiffTable = TableTools.getCommentDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            boolean isWorkspaceAdmin = AdminTools.isWorkspaceAdmin(requestingUser.getId(), new Workspace(workspace));

            StringBuilder sb = new StringBuilder();
            sb.append("SELECT id, gfid, comment, description, insertedat, insertedby FROM " + commentDiffTable
                    + " WHERE type = ?");
            if (!isWorkspaceAdmin) {
                sb.append(" AND insertedby = " + requestingUser.getId());
            }
            sb.append(";");
            System.out.println(sb.toString());

            PreparedStatement ps = jdbcData.prepareStatement(sb.toString());
            ps.setString(1, commentOrLinkType.toString());
            ResultSet rs = ps.executeQuery();

            List<Weblink> weblinks = new ArrayList<>();
            List<Integer> creators = new ArrayList<>();
            while (rs.next()) {
                int id = rs.getInt("id");
                int gfid = rs.getInt("gfid");
                String cmt = rs.getString("comment");
                String insertedat = rs.getString("insertedat");
                String description = rs.getString("description");
                int insertedby = rs.getInt("insertedby");
                Weblink weblink = new Weblink();
                weblink.setId(id);
                weblink.setGfid(gfid);
                weblink.setLink(cmt);
                weblink.setDescription(description);
                weblink.setCreatedat(insertedat);
                // this is the user id but needs to be transformed to a user name
                weblink.setCreator(String.valueOf(insertedby));

                creators.add(insertedby);

                weblinks.add(weblink);
            }

            addCreatorInfo(weblinks, creators, isWorkspaceAdmin);

            return weblinks;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Accept a comment from a comments diff table.
     *
     * @param diffTable
     * @param dataTable
     * @param grpfid
     * @param features
     * @param userId
     */
    public static void acceptCommentOrWeblink(String commentsTable, String diffCommentsTable,
            int gfid, int diffCommentId) {

        QueryCheck.checkTableName(commentsTable);
        QueryCheck.checkDiffTableName(diffCommentsTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String accept = "INSERT INTO " + commentsTable + " (comment,gfid,insertedby,insertedat,type,description) "
                    + "SELECT comment,gfid,insertedby,insertedat,type,description from " + diffCommentsTable
                    + " WHERE gfid = " + gfid + " AND id = " + diffCommentId + ";";

            accept += "DELETE FROM " + diffCommentsTable + " WHERE gfid = " + gfid + " AND id = " + diffCommentId + ";";

            jdbcData.execute(accept);

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void declineCommentOrWeblink(String diffCommentsTable,
            int gfid, int diffCommentId) {

        QueryCheck.checkDiffTableName(diffCommentsTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String decline = "DELETE FROM " + diffCommentsTable + " WHERE gfid = " + gfid + " AND id = " + diffCommentId
                    + ";";
            jdbcData.execute(decline);

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void deleteCommentOrWeblink(String commentsTable, int gfid, int commentOrWeblinkId) {
        QueryCheck.checkTableName(commentsTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String delete = "DELETE FROM " + commentsTable + " WHERE gfid = " + gfid + " AND id = " + commentOrWeblinkId
                    + ";";
            jdbcData.execute(delete);

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    private static List<Weblink> addCreatorInfo(List<Weblink> commentOrWeblinks,
            List<Integer> creatorIds, boolean asWorkspaceAdmin) {

        List<User> creators = UserRepository.getUsersById(creatorIds);
        Map<Integer, User> creatorMap = new HashMap<Integer, User>();
        for (User creator : creators) {
            creatorMap.put(creator.getId(), creator);
        }

        for (Weblink commentOrWeblink : commentOrWeblinks) {
            int userid = Integer.valueOf(commentOrWeblink.getCreator());
            User documentCreator = creatorMap.get(userid);
            String documentCreatorMail = "deleted-user@unknown";
            String documentCreatorName = "deleted-user";
            if (documentCreator != null) {
                documentCreatorMail = documentCreator.getEmail();
                documentCreatorName = ParentDatasetRepository.getCreatorName(documentCreatorMail);
            }

            commentOrWeblink.setCreator(documentCreatorName);
            if (asWorkspaceAdmin) {
                commentOrWeblink.setCreatormail(documentCreatorMail);
            }
        }

        return commentOrWeblinks;
    }
}
