package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import org.geotools.feature.FeatureCollection;

import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.dto.FeatureType;
import de.narimo.georepo.server.layer.GeometryType;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class DatasetRepository extends ParentDatasetRepository {

    public static FeatureCollection getFeatures(int layerId, Integer gfid, boolean asAdmin)
            throws SQLException {

        String dataTableName = TableTools.getDataTableName(layerId);
        QueryCheck.checkTableName(dataTableName);
        String authTableName = TableTools.getAuthTableName(layerId);
        QueryCheck.checkTableName(authTableName);

        Map<String, String> columnsAndTypes = getColumnTypes(dataTableName,
                GeorepoConstants.DEFAULTSCHEMA);

        Map<String, Class> propertyTypes = new HashMap<String, Class>();

        for (String colName : columnsAndTypes.keySet()) {
            String colType = columnsAndTypes.get(colName);
            if (colType.equals("integer")) {
                propertyTypes.put(colName, Integer.class);
            } else if (colType.equals("double precision")) {
                propertyTypes.put(colName, Double.class);
            } else {
                propertyTypes.put(colName, String.class);
            }
        }

        Map<Integer, List<String>> tags = null;
        if (TagRepository.checkTagTableExists(layerId)) {
            if (propertyTypes.containsKey("tags")) {
                System.out.println("Warning: Data table already contains a property 'tags'. "
                        + "Cannot match tags from tags table to the dataset.");
            } else {
                columnsAndTypes.put("tags", "text");
                propertyTypes.put("tags", String[].class);
                tags = TagRepository.getTagsWithGfid(layerId);
            }
        }

        String query = "SELECT ST_AsText(the_geom), * from public." + dataTableName + ";";
        if (gfid != null) {
            query = "SELECT ST_AsText(the_geom), * from public." + dataTableName + " WHERE gfid = " + gfid + ";";
        }
        if (asAdmin && checkAuthTableExists(layerId)) {
            columnsAndTypes.put("modifiedat", "text");
            columnsAndTypes.put("modifiedby", "text");
            propertyTypes.put("modifiedat", String.class);
            propertyTypes.put("modifiedby", String.class);
            query = "SELECT ST_AsText(o.the_geom), o.*, a.modifiedby, a.modifiedat from public." + dataTableName
                    + " AS o LEFT JOIN public." + authTableName + " AS a "
                    + " ON o.grpfid = a.grpfid";
            if (gfid != null) {
                query = "SELECT ST_AsText(o.the_geom), o.*, a.modifiedby, a.modifiedat from public." + dataTableName
                        + " AS o LEFT JOIN public." + authTableName + " AS a "
                        + " ON o.grpfid = a.grpfid"
                        + " WHERE o.gfid = " + gfid;
            }
        }

        return createSimpleFeatures(dataTableName, propertyTypes, columnsAndTypes, query, tags, asAdmin, asAdmin);
    }

    public static int getLayerId(String workspace, String featureType) {

        JDBCConnectionJNDI jdbcMeta = null;
        Integer id = null;
        int count = 0;
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT id from public.datasets WHERE workspace = ? AND featuretype = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setString(1, workspace);
            ps.setString(2, featureType);

            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                id = rs.getInt("id");
                count++;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }

        if (count > 1) {
            throw new NotFoundException(
                    "Could not determine distinct layer id. Please review datasets table consistency for workspace "
                            + workspace + " and feature type " + featureType + "!");
        }
        if (id == null) {
            throw new NotFoundException(
                    "Could not determine layer id for workspace "
                            + workspace + " and feature type " + featureType + "!");
        }
        return id;
    }

    /**
     * Checks, whether a feature with gfid exists for the given layer.
     * 
     * @param layerId
     * @param gfid
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean doesFeatureExist(int layerId, int gfid)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String dataTable = TableTools.getDataTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT count(*) FROM " + dataTable + " WHERE gfid = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            rs.next();
            int count = rs.getInt("count");
            if (count > 0) {
                return true;
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return false;
    }

    public static List<String> getCategories(int layerId, String optCategoryColumn) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String dataTable = TableTools.getDataTableName(layerId);

        List<String> categories = new ArrayList<>();

        String categoryColumn = "category";
        if (optCategoryColumn != null && optCategoryColumn.chars().allMatch(Character::isLetter)) {
            categoryColumn = optCategoryColumn;
        }

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name FROM information_schema.columns WHERE table_name=? and column_name=?;";
            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, dataTable);
            ps0.setString(2, categoryColumn);
            ResultSet rs0 = ps0.executeQuery();
            if (!rs0.next()) {
                return new ArrayList<>();
            }

            String sql = "SELECT DISTINCT " + categoryColumn + " FROM " + dataTable;
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String category = rs.getString(1);
                if (category != null) {
                    categories.add(rs.getString(1));
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return categories;
    }

    public static boolean checkAuthTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String authTable = TableTools.getAuthTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", authTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Retrieves the history datasets of a feature for a given gfid ordered by
     * insertedat date descending (newest first).
     * 
     * @param layerId
     * @param userId
     * @return
     * @throws SQLException
     */
    public static FeatureCollection getDatasetHistory(int layerId, long userId, int gfid) throws SQLException {

        if (!ParentDatasetRepository.checkHistoryTableExists(layerId)) {
            // no history datasets available, since we haven't stored any yet
            return null;
        }

        String histTableName = TableTools.getHistTableName(layerId);
        QueryCheck.checkTableName(histTableName);

        Map<String, String> columnsAndTypes = getColumnTypes(histTableName,
                GeorepoConstants.DEFAULTSCHEMA);

        Map<String, Class> propertyTypes = new HashMap<String, Class>();

        for (String colName : columnsAndTypes.keySet()) {
            String colType = columnsAndTypes.get(colName);
            if (colType.equals("integer")) {
                propertyTypes.put(colName, Integer.class);
            } else if (colType.equals("double precision")) {
                propertyTypes.put(colName, Double.class);
            } else {
                propertyTypes.put(colName, String.class);
            }
        }

        String query = "SELECT ST_AsText(the_geom), *, insertedby as insertedbyuser from public." + histTableName
                + " WHERE gfid = " + gfid + " ORDER BY insertedat asc NULLS FIRST;";
        System.out.println(query);

        // we should only allow retrieving history, so we set addUserInformation=true!
        return createSimpleFeatures(histTableName, propertyTypes, columnsAndTypes, query, null, true, true);
    }

    /**
     * Creates the auth table for the gd_x table
     * 
     * @param authTableName
     * @throws SQLException
     * @throws IOException
     */
    public static void createAuthTable(String authTableName) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        QueryCheck.checkTableName(authTableName);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", authTableName)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + authTableName
                    + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "grpfid integer UNIQUE NOT NULL,"
                    + "diff_grpfid integer UNIQUE,"
                    + "modifiedby integer NOT NULL,"
                    + "modifiedat timestamp without time zone default '1970-01-01 00:00:01' NOT NULL"
                    + ");";
            System.out.println(sql);

            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void createHistoryTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String dataTable = TableTools.getDataTableName(layerId);
        String histTable = TableTools.getHistTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (tableExists(jdbcData, "public", histTable)) {
                return;
            }

            // #378: Add insertedat timestamp as '1970-01-01 00:00:01'
            // so that there is never a null timestamp but we don't
            // get confused with correct timestamps that should come from the _auth table.

            // #379: Do not use IF NOT EXISTS! It will allow subsequent queries to run, even
            // if the initial CREATE TABLE failed because the table does exist.
            String sql = "CREATE TABLE public." + histTable
                    + " (LIKE " + dataTable + " INCLUDING DEFAULTS);"
                    + " ALTER TABLE " + histTable
                    + " DROP COLUMN " + GeorepoConstants.GRPFIDCOLUMN + ","
                    + " DROP COLUMN " + GeorepoConstants.GFIDCOLUMN + ","
                    + " ADD COLUMN " + GeorepoConstants.GRPFIDCOLUMN + " integer NOT NULL," // do not create auto values
                    + " ADD COLUMN " + GeorepoConstants.GFIDCOLUMN + " integer NOT NULL," // do not create auto values
                    + " ADD COLUMN IF NOT EXISTS insertedat timestamp without time zone DEFAULT '1970-01-01 00:00:01',"
                    + " ADD COLUMN IF NOT EXISTS insertedby integer; ";

            // don't add a gfid unique constraint here!

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static List<FeatureType> getLayerTitles(String dataWorkspace, List<String> featureTypes)
            throws SQLException {
        List<FeatureType> layers = new ArrayList<>();
        if (featureTypes.size() == 0) {
            return layers;
        }

        JDBCConnectionJNDI jdbcData = null;
        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT workspace, layer, featuretype, geometrytype FROM datasets WHERE workspace = ? AND featuretype IN (";
            for (int i = 0; i < featureTypes.size(); i++) {
                if (i > 0) {
                    sql += ",";
                }
                sql += "?";
            }
            sql += ");";

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setString(1, dataWorkspace);
            int k = 2;
            for (String featureType : featureTypes) {
                ps.setString(k++, featureType);
            }
            System.out.println(ps);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String featuretype = rs.getString("featuretype");
                String layertitle = rs.getString("layer");
                String geometrytype = rs.getString("geometrytype");

                FeatureType featureType = new FeatureType();
                featureType.setName(featuretype);
                featureType.setTitle(layertitle);

                // don't add layers that don't have a geometry type specified
                // as it will break a client
                if (geometrytype != null) {
                    try {
                        GeometryType gt = GeometryType.valueOf(geometrytype);
                        featureType.setGeometryType(gt.name());
                        layers.add(featureType);
                    } catch (Exception e) {
                        // ignore this layer if valueOf throws Exception
                    }
                }
            }
            return layers;
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void main(String[] args) {
        String geometrytype = "pointx";
        GeometryType gt = GeometryType.valueOf(geometrytype);
    }
}
