package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.NotFoundException;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.api.documents.DocumentType;
import de.narimo.georepo.server.api.documents.FeatureDocument;
import de.narimo.georepo.server.api.workspaces.Workspace;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class ImageOrDocumentRepository {

    /**
     * Creates a table that stores images related to a POI of the layer that this
     * table is related to.
     * 
     * @param layerId
     * @throws SQLException
     * @throws IOException
     */
    public static void createImageTable(int layerId) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        String imgDataTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", imgDataTable)) {
                return;
            }
            System.out.println("Image table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + imgDataTable + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "gfid integer NOT NULL,"
                    + "filename text UNIQUE NOT NULL,"
                    + "description text,"
                    + "type text NOT NULL,"
                    + "insertedby integer NOT NULL,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Creates an image diff table from the structure of an image table. And adds a
     * grpstatus column.
     *
     * @param dataTable
     */
    public static void createImageDiffTable(int layerId) throws SQLException, IOException {
        JDBCConnectionJNDI jdbcData = null;

        String imgDataTable = TableTools.getImageTableName(layerId);
        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", imgDiffTable)) {
                return;
            }
            System.out.println("Diff table does not yet exist for layer " + layerId + ". Creating it...");

            String sql = "CREATE TABLE public." + imgDiffTable + ""
                    + " (LIKE " + imgDataTable + " INCLUDING DEFAULTS);"
                    + " ALTER TABLE " + imgDiffTable
                    + " DROP COLUMN id,"
                    + " ADD COLUMN id SERIAL PRIMARY KEY NOT NULL,"
                    + " ADD COLUMN " + GeorepoConstants.GRPSTATUSCOLUMN + " character varying(1) NOT NULL;";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Checks, whether an image table exists for a given layer id.
     * 
     * @param layerId
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean checkImageTableExists(int layerId) throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", imgTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }

    }

    /**
     * Checks, if a image diff table exist for a given layer id.
     * 
     * @param layerId
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean checkImageDiffTableExists(int layerId) throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", imgDiffTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Inserts a new image and returns its id.
     * 
     * @param layerId
     * @param userId
     * @param gfid
     * @param generatedFilename
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static int insertUnreviewedImage(int layerId, Integer userId, int gfid, String generatedFilename,
            DocumentType documentType, String description)
            throws SQLException, IOException {

        if (userId == null) {
            throw new RuntimeException();
        }

        JDBCConnectionJNDI jdbcData = null;

        String imgDiffTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "INSERT INTO " + imgDiffTable
                    + " (gfid, filename, type, insertedby, grpstatus, description) VALUES (?, ?, ?, ?, ?, ?) RETURNING id;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, generatedFilename);
            ps.setString(3, documentType.toString());
            ps.setInt(4, userId);
            ps.setString(5, "n");
            ps.setString(6, description);

            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                return rs.getInt("id");
            }
            throw new RuntimeException("Could not insert image.");

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * @deprecated does not return image description. Use
     *             getFeatureImagesOrDocuemnts instead.
     * 
     *             Retrieves a list of image names for a feature.
     * 
     * @param layerId
     * @param gfid
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static List<String> getFeatureImages(int layerId, int gfid)
            throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);
        List<String> imageOrDocumentNames = new ArrayList<String>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT filename FROM " + imgTable + " WHERE gfid = ? "
                    + "AND type = 'IMAGE';";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String filename = rs.getString("filename");
                if (filename == null) {
                    throw new InternalError("Invalid file entry for layer=" + layerId + " and gfid=" + gfid + ".");
                }
                imageOrDocumentNames.add(filename);
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return imageOrDocumentNames;
    }

    /**
     * Retrieves a list of documents with related information for each feature.
     * 
     * @param layerId
     * @param gfid
     * @param documentType
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static List<FeatureDocument> getFeatureImagesOrDocuemnts(User requestingUser, String workspace, int layerId,
            int gfid, DocumentType documentType)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String imageOrDocumentsTable = TableTools.getImageTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT filename, description, insertedat, insertedby FROM " + imageOrDocumentsTable
                    + " WHERE gfid = ? AND type = '" + documentType.toString() + "';";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ResultSet rs = ps.executeQuery();

            List<FeatureDocument> documents = new ArrayList<>();
            List<Integer> creatorIds = new ArrayList<>();
            while (rs.next()) {
                String filename = rs.getString("filename");
                String insertedat = rs.getString("insertedat");
                String description = rs.getString("description");
                int insertedby = rs.getInt("insertedby");
                FeatureDocument document = new FeatureDocument();
                document.setName(filename);
                document.setCreatedat(insertedat);
                document.setDescription(description);
                // this is the user id but needs to be transformed to a user name
                document.setCreator(String.valueOf(insertedby));

                creatorIds.add(insertedby);

                documents.add(document);
            }

            addCreatorInfo(documents, creatorIds,
                    AdminTools.isWorkspaceAdmin(requestingUser.getId(), new Workspace(workspace)));

            return documents;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Accept an image from an images diff table.
     *
     * @param imagesTable
     * @param diffImagesTable
     * @param gfid
     * @param diffImageId
     * @throws IOException
     */
    public static String acceptNewImageOrDocument(String imagesTable, String diffImagesTable,
            int gfid, String diffDocumentFilename, DocumentType documentType) throws IOException {

        QueryCheck.checkTableName(imagesTable);
        QueryCheck.checkDiffTableName(diffImagesTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String accept = "INSERT INTO " + imagesTable + " (filename,type,gfid,insertedby,insertedat,description) "
                    + "SELECT filename,type,gfid,insertedby,insertedat,description from " + diffImagesTable
                    + " WHERE gfid = " + gfid + " AND filename = ?"
                    + " AND grpstatus = 'n' AND type=?;";

            accept += "DELETE FROM " + diffImagesTable + " WHERE gfid = " + gfid + " AND filename = ?"
                    + " AND grpstatus = 'n' AND type=?;";

            System.out.println(accept);
            PreparedStatement ps1 = jdbcData.prepareStatement(accept);
            ps1.setString(1, diffDocumentFilename);
            ps1.setString(2, documentType.name());
            ps1.setString(3, diffDocumentFilename);
            ps1.setString(4, documentType.name());
            ps1.execute();

            return diffDocumentFilename;
        } catch (NotFoundException e) {
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static String declineNewImageOrDocument(String imagesTable, String diffImagesTable,
            int gfid, String diffDocumentFilename, DocumentType documentType) throws IOException {

        QueryCheck.checkTableName(imagesTable);
        QueryCheck.checkDiffTableName(diffImagesTable);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String decline = "DELETE FROM " + diffImagesTable + " WHERE gfid = " + gfid + " AND filename = ?"
                    + " AND grpstatus = 'n' AND type=?;";

            System.out.println(decline);
            PreparedStatement ps1 = jdbcData.prepareStatement(decline);
            ps1.setString(1, diffDocumentFilename);
            ps1.setString(2, documentType.name());
            ps1.execute();

            return diffDocumentFilename;
        } catch (NotFoundException e) {
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Check, if there is an image with the given name for the given layer and
     * feature.
     * 
     * @param layerId
     * @param gfid
     * @param filename
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static boolean isImageFromFeature(int layerId, int gfid, String filename, boolean diffFeature)
            throws SQLException, IOException {

        JDBCConnectionJNDI jdbcData = null;

        String imgTable = TableTools.getImageTableName(layerId);
        if (diffFeature) {
            imgTable = TableTools.getImageDiffTableName(layerId);
        }

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT count(*) FROM " + imgTable + " WHERE gfid = ? AND filename = ?;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ps.setInt(1, gfid);
            ps.setString(2, filename);
            ResultSet rs = ps.executeQuery();

            rs.next();
            int count = rs.getInt("count");
            if (count == 1) {
                return true;
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return false;
    }

    public static List<FeatureDocument> getUnreviewedImagesOrDocuments(User requestingUser, String workspace,
            int layerId, DocumentType documentType)
            throws SQLException {

        JDBCConnectionJNDI jdbcData = null;
        String diffImageOrDocumentsTable = TableTools.getImageDiffTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            boolean isWorkspaceAdmin = AdminTools.isWorkspaceAdmin(requestingUser.getId(), new Workspace(workspace));

            StringBuilder sb = new StringBuilder();
            sb.append("SELECT filename, gfid, description, insertedat, insertedby FROM " + diffImageOrDocumentsTable
                    + " WHERE type = '" + documentType.toString() + "'");
            if (!isWorkspaceAdmin) {
                sb.append(" AND insertedby = " + requestingUser.getId());
            }
            sb.append(";");
            System.out.println(sb.toString());

            PreparedStatement ps = jdbcData.prepareStatement(sb.toString());
            ResultSet rs = ps.executeQuery();

            List<FeatureDocument> images = new ArrayList<>();
            List<Integer> creators = new ArrayList<>();
            while (rs.next()) {
                int insertedby = rs.getInt("insertedby");
                FeatureDocument image = new FeatureDocument();
                image.setName(rs.getString("filename"));
                image.setGfid(rs.getInt("gfid"));
                image.setCreatedat(rs.getString("insertedat"));
                image.setDescription(rs.getString("description"));
                // this is the user id but needs to be transformed to a user name
                image.setCreator(String.valueOf(insertedby));

                creators.add(insertedby);

                images.add(image);
            }

            addCreatorInfo(images, creators, isWorkspaceAdmin);

            return images;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    private static List<FeatureDocument> addCreatorInfo(List<FeatureDocument> imagesOrDocuments,
            List<Integer> creatorIds, boolean asWorkspaceAdmin) {

        List<User> creators = UserRepository.getUsersById(creatorIds);
        Map<Integer, User> creatorMap = new HashMap<Integer, User>();
        for (User creator : creators) {
            creatorMap.put(creator.getId(), creator);
        }

        for (FeatureDocument document : imagesOrDocuments) {
            int userid = Integer.valueOf(document.getCreator());
            User documentCreator = creatorMap.get(userid);
            String documentCreatorMail = "deleted-user@unknown";
            String documentCreatorName = "deleted-user";
            if (documentCreator != null) {
                documentCreatorMail = documentCreator.getEmail();
                documentCreatorName = ParentDatasetRepository.getCreatorName(documentCreatorMail);
            }

            document.setCreator(documentCreatorName);
            if (asWorkspaceAdmin) {
                document.setCreatormail(documentCreatorMail);
            }
        }
        return imagesOrDocuments;
    }

}
