package de.narimo.georepo.server.repository;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Geometry;
import org.opengis.feature.simple.SimpleFeature;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.GeorepoConstants;
import de.narimo.georepo.server.appconfig.GeorepoConfigFeaturePropertyStore;
import de.narimo.georepo.server.layer.ColumnProperties;
import de.narimo.georepo.server.layer.FeatureBuilder;
import de.narimo.georepo.server.layer.GeometryType;
import de.narimo.georepo.server.tools.GeometryTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class ParentDatasetRepository {

    /**
     * @see https://stackoverflow.com/a/24089729/3271380
     * @return
     * @throws SQLException
     */
    public static boolean tableExists(JDBCConnectionJNDI jdbc, String schemaName, String tableName)
            throws SQLException {

        String sql = "SELECT EXISTS ("
                + "     SELECT FROM pg_catalog.pg_class c"
                + "     JOIN   pg_catalog.pg_namespace n ON n.oid = c.relnamespace"
                + "     WHERE  n.nspname = ?"
                + "     AND    c.relname = ?);";
//        System.out.println(sql);

        PreparedStatement ps = jdbc.prepareStatement(sql);
        ps.setString(1, schemaName);
        ps.setString(2, tableName);

        ResultSet rs = jdbc.executePreparedQuery(ps);
        rs.next();
        if (rs.getBoolean("exists") == Boolean.FALSE) {
            System.out.println("TABLE DOES NOT EXIST");
            return false;
        }
        System.out.println("TABLE EXISTS");
        return true;
    }

    /**
     * @deprecated use getColumnFeatureTypes instead
     * @param tableName
     * @param schemaName
     * @return
     */
    public static Map<String, String> getColumnTypes(String tableName, String schemaName) {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name, data_type FROM information_schema.columns " + "WHERE table_schema = ? "
                    + "AND table_name = ?;";

            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, schemaName);
            ps0.setString(2, tableName);

            List<String> reservedColumnNames = new ArrayList<>();
            reservedColumnNames.add(GeorepoConstants.GRPFIDCOLUMN);
            // reservedColumnNames.add(GeorepoConstants.GFIDCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GEOMETRYCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GRPSTATUSCOLUMN);

            Map<String, String> columnNames = new HashMap<>();

            ResultSet rs0 = ps0.executeQuery();
            while (rs0.next()) {
                String columnName = rs0.getString("column_name");
                if (!reservedColumnNames.contains(columnName)) {
                    columnNames.put(columnName, rs0.getString("data_type"));
                }
            }
            return columnNames;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static List<ColumnProperties> getColumnFeatureTypes(String tableName, String schemaName) {
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT column_name, data_type FROM information_schema.columns " + "WHERE table_schema = ? "
                    + "AND table_name   = ?;";

            PreparedStatement ps0 = jdbcData.prepareStatement(sql0);
            ps0.setString(1, schemaName);
            ps0.setString(2, tableName);

            List<String> reservedColumnNames = new ArrayList<>();
            reservedColumnNames.add(GeorepoConstants.GRPFIDCOLUMN);
            // reservedColumnNames.add(GeorepoConstants.GFIDCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GEOMETRYCOLUMN);
            reservedColumnNames.add(GeorepoConstants.GRPSTATUSCOLUMN);

            List<ColumnProperties> featureType = new ArrayList<>();

            ResultSet rs0 = ps0.executeQuery();
            while (rs0.next()) {
                String columnName = rs0.getString("column_name");
                if (!reservedColumnNames.contains(columnName)) {
                    ColumnProperties columnProperties = new ColumnProperties();
                    columnProperties.setName(rs0.getString("column_name"));
                    columnProperties.setType(rs0.getString("data_type"));
                    featureType.add(columnProperties);
                }
            }
            return featureType;
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return null;
    }

    public static FeatureCollection createSimpleFeatures(String dataTableName, Map<String, Class> propertyTypes,
            Map<String, String> columnsAndTypes, String query, Map<Integer, List<String>> tags,
            boolean addUserInformation, boolean asWorkspaceAdmin) {

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            GeometryType geometryType = detectGeometryType(dataTableName);
            if (geometryType == null) {
                System.out.println("Detected geometry type is: " + geometryType);
            }

            List<SimpleFeature> simpleFeatures = new ArrayList<>();
            List<Integer> modificatorIds = new ArrayList<>();
            List<GeorepoConfigFeaturePropertyStore> propertyStore = new ArrayList<>();

            String userSelectorColumn = null;
            if (query.contains("modifiedby")) {
                userSelectorColumn = "modifiedby";
            } else if (query.contains("insertedby")) {
                userSelectorColumn = "insertedby";
            }

            System.out.println(query);
            ResultSet rs = jdbcData.executeQuery(query);

            while (rs.next()) {
                int gfid = rs.getInt("gfid");
                String wkt = rs.getString("st_astext");

                Geometry geometry = null;
                if (wkt != null) {
                    geometry = GeometryTools.createGeometry(wkt);
                }

                Map<String, Object> properties = new HashMap<>();
                for (String colName : columnsAndTypes.keySet()) {
                    if (colName.equals("tags") && tags != null) {
                        if (tags.get(gfid) != null && tags.get(gfid).size() > 0) {
                            properties.put(colName, tags.get(gfid));
                        }
                    } else {
                        properties.put(colName, rs.getString(colName));
                    }
                }

                // this contains the user id but needs to be transformed to a user name below
                if (userSelectorColumn != null) {
                    modificatorIds.add(rs.getInt(userSelectorColumn));
                }

                GeorepoConfigFeaturePropertyStore featureStore = new GeorepoConfigFeaturePropertyStore();
                featureStore.setGeometry(geometry);
                featureStore.setProperties(properties);
                propertyStore.add(featureStore);
            }

            if (addUserInformation && userSelectorColumn != null) {
                // modificators that we get from the database
                // excludes users that have been disabled in the meantime!
                List<User> modificators = UserRepository.getUsersById(modificatorIds);
                Map<Integer, User> modificatorMap = new HashMap<Integer, User>();
                for (User modificator : modificators) {
                    modificatorMap.put(modificator.getId(), modificator);
                }

                for (GeorepoConfigFeaturePropertyStore featureStore : propertyStore) {
                    try {
                        Object modifyingUser = featureStore.getProperties().get(userSelectorColumn);
                        if (modifyingUser != null) {
                            String modifiedByAsInt = (String) modifyingUser;
                            Integer userid = Integer.valueOf(modifiedByAsInt);
                            if (userid != null) {
                                User featureModificator = modificatorMap.get(userid);
                                String featureModificatorMail = "deleted-user@unknown";
                                String featureModificatorName = "deleted-user";
                                if (featureModificator != null) {
                                    featureModificatorMail = featureModificator.getEmail();
                                    featureModificatorName = getCreatorName(featureModificatorMail);
                                }

                                featureStore.getProperties().put("creator", featureModificatorName);
                                if (asWorkspaceAdmin) {
                                    featureStore.getProperties().put("creatormail", featureModificatorMail);
                                }
                            }
                            featureStore.getProperties().put(userSelectorColumn, null);
                        }
                    } catch (NumberFormatException | ClassCastException e) {
                        // Do nothing, ignore this feature
                        System.out.println("Error on reading inserting/ modifying user. Setting null.");
                    }

                    propertyTypes.put("creator", String.class);
                    if (asWorkspaceAdmin) {
                        propertyTypes.put("creatormail", String.class);
                    }
                }
            }

            FeatureBuilder fb = new FeatureBuilder(geometryType, propertyTypes);
            for (GeorepoConfigFeaturePropertyStore featureStore : propertyStore) {
                SimpleFeature simpleFeature = fb.createSimpleFeature(featureStore.getGeometry(),
                        featureStore.getProperties());
                simpleFeatures.add(simpleFeature);
            }
            return fb.createFeatureCollection(simpleFeatures);

        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Intends to detect the geometry type from the the_geom colum of the first
     * entry of the data table.
     * 
     * @param dataTableName
     * @return
     * @throws Exception
     */
    public static GeometryType detectGeometryType(String dataTableName) {
        QueryCheck.checkTableName(dataTableName);

        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String s = "SELECT ST_GeometryType(ST_AsText(the_geom)) from public." + dataTableName
                    + " WHERE the_geom IS NOT NULL limit 1;";
            System.out.println(s);

            ResultSet rs = jdbcData.executeQuery(s);

            if (rs.next()) {
                String geomType = rs.getString("st_geometrytype");
                if (geomType == null) {
                    return null;
                } else if (geomType.equals("ST_Point")) {
                    return GeometryType.POINT;
                } else if (geomType.equals("ST_LineString")) {
                    return GeometryType.LINESTRING;
                } else if (geomType.equals("ST_Polygon")) {
                    return GeometryType.POLYGON;
                }
            }
            return null;

        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Returns the first part of an email before the @ sign.
     * 
     * @param creatorMail
     * @return
     */
    public static String getCreatorName(String creatorMail) {
        if (creatorMail.indexOf('@') == -1) {
            System.out.println("WARNING: Invalid mail to determine creator from: " + creatorMail);
            return null;
        }
        return creatorMail.substring(0, creatorMail.indexOf('@'));
    }

    public static boolean checkHistoryTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String histTable = TableTools.getHistTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", histTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }
}
