package de.narimo.georepo.server.repository;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.geocore.ws.repository.UserRepository;
import de.narimo.georepo.server.api.tags.GeorepoTag;
import de.narimo.georepo.server.api.workspaces.Workspace;
import de.narimo.georepo.server.tools.AdminTools;
import de.narimo.georepo.server.tools.QueryCheck;
import de.narimo.georepo.server.tools.TableTools;

public class TagRepository extends ParentDatasetRepository {

    public static List<String> getTags(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;
        String tagTable = TableTools.getTagTableName(layerId);
        List<String> tags = new ArrayList<>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT DISTINCT tag FROM " + tagTable + " WHERE grpstatus is null;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String tag = rs.getString("tag");
                if (tag != null) {
                    tags.add(tag);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static List<GeorepoTag> getUnreviewedTags(int layerId, String workspace, User requestingUser)
            throws SQLException {
        JDBCConnectionJNDI jdbcData = null;
        String tagTable = TableTools.getTagTableName(layerId);
        List<GeorepoTag> tags = new ArrayList<>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            boolean isWorkspaceAdmin = AdminTools.isWorkspaceAdmin(requestingUser.getId(), new Workspace(workspace));

            StringBuilder sb = new StringBuilder();
            sb.append("SELECT DISTINCT id, gfid, tag, grpstatus, insertedby, insertedat FROM " + tagTable
                    + " WHERE grpstatus is not null ");
            if (!isWorkspaceAdmin) {
                sb.append(" AND insertedby = " + requestingUser.getId());
            }
            sb.append(" ORDER BY id, gfid, tag, grpstatus, insertedby, insertedat;");
            System.out.println(sb.toString());

            PreparedStatement ps = jdbcData.prepareStatement(sb.toString());
            ResultSet rs = ps.executeQuery();

            List<Integer> creators = new ArrayList<>();
            while (rs.next()) {
                GeorepoTag tag = new GeorepoTag();
                tag.setId(rs.getInt("id"));
                tag.setGfid(rs.getInt("gfid"));
                tag.setTag(rs.getString("tag"));
                tag.setGrpstatus(rs.getString("grpstatus"));
                tag.setCreatedat(rs.getString("insertedat"));

                int insertedby = rs.getInt("insertedby");
                // this is the user id but needs to be transformed to a user name
                tag.setCreator(String.valueOf(insertedby));
                creators.add(insertedby);

                tags.add(tag);
            }

            addCreatorInfo(tags, creators, isWorkspaceAdmin);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static Map<Integer, List<String>> getTagsWithGfid(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;
        String tagTable = TableTools.getTagTableName(layerId);
        Map<Integer, List<String>> tags = new HashMap<>();

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql = "SELECT DISTINCT gfid, tag FROM " + tagTable + " WHERE grpstatus is null ORDER BY gfid, tag;";
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String tag = rs.getString("tag");
                if (tag != null) {
                    int gfid = rs.getInt("gfid");
                    if (gfid != 0) {
                        List<String> tagList = tags.get(gfid);
                        if (tagList == null) {
                            tagList = new ArrayList<>();
                        }
                        tagList.add(tag);
                        tags.put(gfid, tagList);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static boolean checkTagTableExists(int layerId) throws SQLException {

        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", tagTable)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    /**
     * Add one or multiple tags for a gfid on a layer.
     * 
     * @param layerId
     * @param gfid
     * @param tags
     * @return
     * @throws SQLException
     * @throws IOException
     */
    public static List<String> addNewTagsSuggestion(int layerId, int userId, int gfid, List<String> tags)
            throws SQLException {
        if (tags == null || tags.isEmpty()) {
            throw new IllegalArgumentException("Tags may not be empty.");
        }
        JDBCConnectionJNDI jdbcData = null;
        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            StringBuilder sql = new StringBuilder();
            sql.append("INSERT INTO " + tagTable + "(gfid, tag, grpstatus, insertedby) VALUES");

            for (int cnt = 0; cnt < tags.size(); cnt++) {
                if (cnt > 0) {
                    sql.append(",");
                }
                sql.append("(" + gfid + ",?,'n',?)");
            }
            sql.append(";");
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql.toString());
            int cnt = 1;
            for (String tag : tags) {
                ps.setString(cnt * 2 - 1, tag);
                ps.setInt(cnt * 2, userId);
                cnt++;
            }
            ps.executeUpdate();

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    /**
     * Mark one or more tags as deleted.
     * 
     * @param layerId
     * @param gfid
     * @param tags
     * @return
     * @throws SQLException
     */
    public static List<String> addDeleteTagsSuggestion(int layerId, int userId, int gfid, List<String> tags)
            throws SQLException {
        if (tags == null || tags.isEmpty()) {
            throw new IllegalArgumentException("Tags may not be empty.");
        }
        JDBCConnectionJNDI jdbcData = null;
        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            StringBuilder sql = new StringBuilder();
            sql.append("INSERT INTO " + tagTable + "(gfid, tag, grpstatus, insertedby) VALUES");

            for (int cnt = 0; cnt < tags.size(); cnt++) {
                if (cnt > 0) {
                    sql.append(",");
                }
                sql.append("(" + gfid + ",?,'d',?)");
            }
            sql.append(";");
            System.out.println(sql);

            PreparedStatement ps = jdbcData.prepareStatement(sql.toString());
            int cnt = 1;
            for (String tag : tags) {
                ps.setString(cnt * 2 - 1, tag);
                ps.setInt(cnt * 2, userId);
                cnt++;
            }
            ps.executeUpdate();

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
        return tags;
    }

    public static void createTagTable(int layerId) throws SQLException {
        JDBCConnectionJNDI jdbcData = null;

        String tagTable = TableTools.getTagTableName(layerId);

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            if (ParentDatasetRepository.tableExists(jdbcData, "public", tagTable)) {
                return;
            }

            String sql = "CREATE TABLE IF NOT EXISTS public." + tagTable + "("
                    + "id SERIAL PRIMARY KEY NOT NULL,"
                    + "gfid integer NOT NULL,"
                    + "tag text NOT NULL,"
                    + "grpstatus text,"
                    + "insertedby integer,"
                    + "insertedat timestamp without time zone DEFAULT timezone('utc'::text, now()) NOT NULL"
                    + ");";

            System.out.println(sql);
            jdbcData.execute(sql);

        } catch (Exception e) {
            e.printStackTrace();
            throw new SQLException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void acceptTag(String tagTable, int gfid, int tagId) {
        QueryCheck.checkTableName(tagTable);
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");

            String sql0 = "SELECT tag, grpstatus FROM public." + tagTable + " WHERE gfid=" + gfid + " AND id=" + tagId;
            System.out.println(sql0);
            ResultSet rs = jdbcData.executeQuery(sql0);
            if (!rs.next()) {
                throw new IllegalArgumentException("No tag suggestion to process with the given id.");
            }
            String tagname = rs.getString("tag");
            char grpstatus = rs.getString("grpstatus").charAt(0);

            String accept;
            if (grpstatus == 'd') {
                accept = "DELETE FROM public." + tagTable + " WHERE gfid=" + gfid + " AND tag=?";
            } else if (grpstatus == 'n') {
                // Delete all other occurrences of the same tag before accepting the new tag
                // suggestion
                accept = "DELETE FROM public." + tagTable
                        + " WHERE gfid=" + gfid + " AND tag=? AND id != " + tagId + ";"
                        + "UPDATE public." + tagTable + " SET grpstatus=null WHERE gfid=" + gfid + " AND id=" + tagId
                        + ";";
            } else {
                throw new RuntimeException(
                        "Expression " + grpstatus + " not allowed for tag update suggestion.");
            }

            System.out.println(accept);
            PreparedStatement ps = jdbcData.prepareStatement(accept);
            ps.setString(1, tagname);
            ps.executeUpdate();
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    public static void declineTag(String tagTable, int gfid, int tagId) {
        QueryCheck.checkTableName(tagTable);
        JDBCConnectionJNDI jdbcData = null;

        try {
            jdbcData = new JDBCConnectionJNDI("jdbc/georepoDataResource");
            String accept = "DELETE FROM " + tagTable + " WHERE gfid=" + gfid + " AND id=" + tagId + ";";
            jdbcData.execute(accept);
        } catch (SQLException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } finally {
            if (jdbcData != null) {
                jdbcData.closeAll();
                jdbcData = null;
            }
        }
    }

    private static List<GeorepoTag> addCreatorInfo(List<GeorepoTag> tags,
            List<Integer> creatorIds, boolean asWorkspaceAdmin) {

        List<User> creators = UserRepository.getUsersById(creatorIds);
        Map<Integer, User> creatorMap = new HashMap<Integer, User>();
        for (User creator : creators) {
            creatorMap.put(creator.getId(), creator);
        }

        for (GeorepoTag tag : tags) {
            int userid = Integer.valueOf(tag.getCreator());
            User documentCreator = creatorMap.get(userid);
            String documentCreatorMail = "deleted-user@unknown";
            String documentCreatorName = "deleted-user";
            if (documentCreator != null) {
                documentCreatorMail = documentCreator.getEmail();
                documentCreatorName = ParentDatasetRepository.getCreatorName(documentCreatorMail);
            }

            tag.setCreator(documentCreatorName);
            if (asWorkspaceAdmin) {
                tag.setCreatormail(documentCreatorMail);
            }
        }

        return tags;
    }
}
