package de.narimo.georepo.server.repository;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import de.narimo.commons.dto.User;
import de.narimo.commons.jdbc.JDBCConnectionJNDI;
import de.narimo.georepo.server.api.workspaces.Workspace;
import de.narimo.georepo.server.tools.WorkspacePermissions;

public class WorkspaceRepository {

    public static void changeWorkspacePermission(int userid, String workspace, String permission)
            throws SQLException {

        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");
            jdbcMeta.setAutoCommit(false);

            String delSQL = "DELETE FROM workspaces WHERE userid=? AND workspace=?;";
            PreparedStatement ps0 = jdbcMeta.prepareStatement(delSQL);
            ps0.setInt(1, userid);
            ps0.setString(2, workspace);
            ps0.executeUpdate();
            System.out.println(ps0.toString());
            if (!permission.equals(WorkspacePermissions.noPermission)) {

                // Set the new permission, if any
                String sql = "INSERT INTO workspaces (userid, workspace, rights) VALUES (?,?,?)";
                PreparedStatement ps1 = jdbcMeta.prepareStatement(sql);
                ps1.setInt(1, userid);
                ps1.setString(2, workspace);
                ps1.setString(3, permission);
                ps1.execute();
                System.out.println(ps1.toString());
            }

            jdbcMeta.commit();

        } catch (Exception e) {
            e.printStackTrace();
            jdbcMeta.rollback();
            throw new RuntimeException(
                    "Could not set workspace permissions for workspace " + workspace + " and user " + userid + ".");
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    public static int getWorkspaceCount(int userId, Workspace workspace, String permission) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT count(*) FROM workspaces w WHERE " + "w.userid = ? AND " + "w.workspace = ? AND "
                    + "w.rights LIKE ?;";
            System.out.println(sql1);

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setInt(1, userId);
            ps.setString(2, workspace.getName());
            ps.setString(3, "%" + permission + "%");

            ResultSet rs1 = ps.executeQuery();
            rs1.next();
            return rs1.getInt("count");

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return 0;
    }

    public static List<Integer> getWorkspaceAdminIds(String workspace) {
        JDBCConnectionJNDI jdbcMeta = null;
        List<Integer> adminIds = new ArrayList<Integer>();
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String permission = WorkspacePermissions.adminPermission;
            String sql1 = "SELECT userid FROM workspaces w WHERE " + "w.workspace = ? AND " + "w.rights LIKE ?;";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setString(1, workspace);
            ps.setString(2, "%" + permission + "%");

            ResultSet rs1 = ps.executeQuery();
            while (rs1.next()) {
                adminIds.add(rs1.getInt("userid"));
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return adminIds;
    }

    /**
     * Returns the type of permission for a specific workspace.
     *
     * @param userId
     * @param workspace
     * @return
     */
    public static String getWorkspacePermissionType(int userId, String workspace) {
        JDBCConnectionJNDI jdbcMeta = null;
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT rights FROM workspaces w WHERE " + "w.workspace = ? AND " + "w.userid = ? LIMIT 1;";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql1);
            ps.setString(1, workspace);
            ps.setInt(2, userId);

            ResultSet rs = ps.executeQuery();

            String rights = WorkspacePermissions.noPermission;

            int count = 0;
            while (rs.next()) {
                count++;
                rights = rs.getString("rights");
            }
            if (count > 1) {
                throw new IllegalArgumentException("Inconsistent entries for workspace permissions. " + count
                        + " different permissions exist.");
            }
            return rights;
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    /**
     * Confirm that a given secrect conforms with the server side workspace secret.
     * Used e.g. to authorize workspace registration.
     *
     * @param workspace
     * @param workspaceSecret
     * @return
     */
    public static boolean isWorkspaceSecretValid(String workspace, String workspaceSecret) {
        JDBCConnectionJNDI jdbcMeta = null;
        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql1 = "SELECT * FROM workspacesecrets w WHERE " + "w.workspace = ? AND w.secret = ?;";

            PreparedStatement ps0 = jdbcMeta.prepareStatement(sql1);
            ps0.setString(1, workspace);
            ps0.setString(2, workspaceSecret);
            ResultSet rs = ps0.executeQuery();

            if (rs.next()) {
                return true;
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    public static List<Workspace> getWorkspaces(int userId, boolean includeDiffWorkspaces) {
        JDBCConnectionJNDI jdbcMeta = null;

        List<Workspace> workspaces = new ArrayList<>();

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT distinct(workspace), rights FROM workspaces WHERE userid= ? "
                    + "AND (workspace IN(SELECT dataworkspace FROM workspaceowners) ";

            if (includeDiffWorkspaces) {
                sql += "OR workspace IN(SELECT diffworkspace FROM workspaceowners) ";
            }
            sql += ") ORDER BY workspace ASC;";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setInt(1, userId);

            ResultSet rs = ps.executeQuery();
            while (rs.next()) {
                Workspace workspace = new Workspace();
                workspace.setName(rs.getString("workspace"));
                workspace.setPermission(WorkspacePermissions.getReadablePermission(rs.getString("rights")));
                workspaces.add(workspace);
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalError();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return workspaces;
    }

    /**
     * Does the user has an allowedworkspacecount counter larger than workspaces
     * that he has already created?
     * 
     * @param user
     * @return
     */
    public static boolean canUserCreateWorkspace(User user) {
        JDBCConnectionJNDI jdbc = null;
        JDBCConnectionJNDI jdbcMeta = null;
        try {
            jdbc = new JDBCConnectionJNDI("jdbc/georepoAuthDatasource");
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String idSql = "SELECT allowedworkspacecount FROM users WHERE id=? AND enabled=true;";
            System.out.println(idSql);

            PreparedStatement ps = jdbc.prepareStatement(idSql);
            ps.setLong(1, user.getId());
            ResultSet rs = ps.executeQuery();

            int allowedWorkspaceCount = 0;
            if (rs.next()) {
                allowedWorkspaceCount = rs.getInt("allowedworkspacecount");
            }

            String sql = "SELECT count(*) FROM workspaceowners WHERE createdby=" + user.getId() + ";";
            System.out.println(sql);
            PreparedStatement ps1 = jdbcMeta.prepareStatement(sql);
            ResultSet rs1 = ps1.executeQuery();

            if (rs1.next()) {
                Integer existingWorkspaces = rs1.getInt("count");
                return allowedWorkspaceCount - existingWorkspaces > 0;
            }

        } catch (Exception e) {
            System.out.println("Could not determine workspace creation permissions for user " + user.getId() + ".");
            e.printStackTrace();
        } finally {
            if (jdbc != null) {
                jdbc.closeAll();
                jdbc = null;
            }
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
        return false;
    }

    public static boolean isDiffWorkspace(String workspace) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            // check if the given data workspace is a diff workspace actually
            String sql2 = "SELECT count(*) FROM workspaceowners WHERE diffworkspace=?";
            System.out.println(sql2);

            PreparedStatement ps2 = jdbcMeta.prepareStatement(sql2);
            ps2.setString(1, workspace);

            ResultSet rs2 = ps2.executeQuery();
            if (rs2.next()) {
                int diffWorkspaceCount = rs2.getInt("count");
                if (diffWorkspaceCount > 0) {
                    return true;
                }
            }
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    /**
     * Add a new entry for a data workspace to workspaceowners.
     * 
     * @param creatingUser
     * @param dataWorkspace
     * @param diffWorkspace
     */
    public static void addDataWorkspaceEntry(User creatingUser, Workspace dataWorkspace) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql0 = "SELECT * FROM workspaceowners WHERE dataworkspace=?";
            PreparedStatement ps0 = jdbcMeta.prepareStatement(sql0);
            ps0.setString(1, dataWorkspace.getName());
            System.out.println(sql0);

            ResultSet rs0 = ps0.executeQuery();
            int dataWsCount = 0;
            while (rs0.next()) {
                dataWsCount++;
            }
            System.out.println("Data workspace (" + dataWorkspace.getName() + ") count: " + dataWsCount);

            if (dataWsCount >= 1) {
                throw new IllegalStateException(
                        "ERROR: Inconsistencies in table workspaceowners found. One or multiple entries already exist for data workspace "
                                + dataWorkspace + ".");
            }

            if (dataWsCount < 1) {
                // No entry for data workspace yet, add a fresh entry
                String sql2 = "INSERT INTO workspaceowners (dataworkspace, createdby) VALUES (?,?)";
                System.out.println(sql2);

                PreparedStatement ps2 = jdbcMeta.prepareStatement(sql2);
                ps2.setString(1, dataWorkspace.getName());
                ps2.setInt(2, creatingUser.getId());

                ps2.execute();
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }

    /**
     * Add an entry for a diff workspace to table workspaceowners. This is new from
     * v2.8 enforcing a 1:1 relation for workspaces and diff workspaces.
     * 
     * @param dataWorkspace
     * @param diffWorkspace
     */
    public static void addDiffWorkspaceEntry(User creatingUser, Workspace dataWorkspace, Workspace diffWorkspace) {
        JDBCConnectionJNDI jdbcMeta = null;

        try {
            jdbcMeta = new JDBCConnectionJNDI("jdbc/georepoMetaResource");

            String sql = "SELECT * FROM workspaceowners WHERE diffworkspace=?";

            PreparedStatement ps = jdbcMeta.prepareStatement(sql);
            ps.setString(1, diffWorkspace.getName());
            System.out.println(sql);

            ResultSet rs = ps.executeQuery();
            int diffCount = 0;
            while (rs.next()) {
                diffCount++;
            }
            System.out.println("Diff workspace (" + diffWorkspace.getName() + ") count: " + diffCount);

            if (diffCount == 1) {
                // yeix, we already have a diff workspace, return
                System.out.println(
                        "WARNING: Cannot add entry for diff workspace to table workspaceowners. Relation for diffworkspace "
                                + diffWorkspace.getName() + " already exists.");
                return;
            }

            String sql0 = "SELECT * FROM workspaceowners WHERE dataworkspace=?";
            System.out.println(sql0);

            PreparedStatement ps0 = jdbcMeta.prepareStatement(sql0);
            ps0.setString(1, dataWorkspace.getName());

            ResultSet rs0 = ps0.executeQuery();
            int dataWsCount = 0;
            while (rs0.next()) {
                dataWsCount++;
            }
            System.out.println("Data workspace (" + dataWorkspace.getName() + ") count: " + dataWsCount);

            if (dataWsCount > 1) {
                throw new IllegalStateException(
                        "ERROR: Inconsistencies in table workspaceowners found. Multiple entries for data workspace "
                                + dataWorkspace + " exist.");
            }

            if (dataWsCount == 1) {
                // Entry for data workspace already exists, connect the diff workspace
                String sql1 = "UPDATE workspaceowners SET diffworkspace=? WHERE dataworkspace=?";
                System.out.println(sql1);

                PreparedStatement ps1 = jdbcMeta.prepareStatement(sql1);
                ps1.setString(1, diffWorkspace.getName());
                ps1.setString(2, dataWorkspace.getName());

                ps1.execute();

            } else {
                // No entry for data workspace yet, add a fresh entry
                String sql2 = "INSERT INTO workspaceowners (dataworkspace, diffworkspace, createdby) VALUES (?,?,?)";
                System.out.println(sql2);

                PreparedStatement ps2 = jdbcMeta.prepareStatement(sql2);
                ps2.setString(1, dataWorkspace.getName());
                ps2.setString(2, diffWorkspace.getName());
                ps2.setInt(3, creatingUser.getId());

                ps2.execute();
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        } finally {
            if (jdbcMeta != null) {
                jdbcMeta.closeAll();
                jdbcMeta = null;
            }
        }
    }
}
