package de.narimo.georepo.server.tools;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.Base64;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.ServletContext;

public class AESGenerator {

    String password;
    String salt;
    SecretKey key;

    public AESGenerator(ServletContext ctx) throws NoSuchAlgorithmException, InvalidKeySpecException {
        password = ctx.getInitParameter("aes_password");
        salt = ctx.getInitParameter("aes_salt");
        this.key = getKeyFromPassword(password, salt);
    }

    public AESGenerator(String password, String salt) throws NoSuchAlgorithmException, InvalidKeySpecException {
        this.password = password;
        this.salt = salt;
        this.key = getKeyFromPassword(password, salt);
    }

    private static String encryptPw(String algorithm, String input, SecretKey key,
            IvParameterSpec iv) throws NoSuchPaddingException, NoSuchAlgorithmException,
            InvalidAlgorithmParameterException, InvalidKeyException,
            BadPaddingException, IllegalBlockSizeException {

        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(Cipher.ENCRYPT_MODE, key, iv);
        byte[] cipherText = cipher.doFinal(input.getBytes());
        return Base64.getEncoder()
                .encodeToString(cipherText);
    }

    private static String decryptPw(String algorithm, String cipherText, SecretKey key,
            IvParameterSpec iv) throws NoSuchPaddingException, NoSuchAlgorithmException,
            InvalidAlgorithmParameterException, InvalidKeyException,
            BadPaddingException, IllegalBlockSizeException {

        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(Cipher.DECRYPT_MODE, key, iv);
        byte[] plainText = cipher.doFinal(Base64.getDecoder()
                .decode(cipherText));
        return new String(plainText);
    }

    private static byte[] getByteArray() { byte[] iv = new byte[16]; new SecureRandom().nextBytes(iv); return iv; }

    private static IvParameterSpec generateIv(byte[] iv) { return new IvParameterSpec(iv); }

    private static SecretKey getKeyFromPassword(String password, String salt)
            throws NoSuchAlgorithmException, InvalidKeySpecException {

        SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA256");
        KeySpec spec = new PBEKeySpec(password.toCharArray(), salt.getBytes(), 65536, 256);
        SecretKey secret = new SecretKeySpec(factory.generateSecret(spec)
                .getEncoded(), "AES");
        return secret;
    }

    /**
     * Encrypts an input text, returning the encrypted string, prepended by the IV.
     * 
     * @param text
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     * @throws InvalidKeyException
     * @throws NoSuchPaddingException
     * @throws InvalidAlgorithmParameterException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public String encrypt(String text) throws NoSuchAlgorithmException, InvalidKeySpecException, InvalidKeyException,
            NoSuchPaddingException, InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException {
        byte[] iv = getByteArray();
        String ivAsText = new String(iv, StandardCharsets.ISO_8859_1);
        IvParameterSpec ivParameterSpec = generateIv(iv);
        SecretKey key = getKeyFromPassword(password, salt);
        String encrypted = encryptPw("AES/CBC/PKCS5Padding", text, key, ivParameterSpec);
        return ivAsText + encrypted;
    }

    /**
     * Decrypts an encrypted string. Assumes that the encrypted string is prepended
     * by its IV.
     * 
     * @param encryptedText
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     * @throws InvalidKeyException
     * @throws NoSuchPaddingException
     * @throws InvalidAlgorithmParameterException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public String decrypt(String encryptedText)
            throws NoSuchAlgorithmException, InvalidKeySpecException, InvalidKeyException, NoSuchPaddingException,
            InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException {
        byte[] iv = encryptedText.substring(0, 16).getBytes(StandardCharsets.ISO_8859_1);
        IvParameterSpec ivParameterSpec = generateIv(iv);
        SecretKey key = getKeyFromPassword(password, salt);
        String encryptedpw = encryptedText.substring(16, encryptedText.length());
        return decryptPw("AES/CBC/PKCS5Padding", encryptedpw, key,
                ivParameterSpec);
    }

    public static void main(String[] args) throws InvalidKeyException, NoSuchPaddingException,
            InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException,
            NoSuchAlgorithmException, InvalidKeySpecException, UnsupportedEncodingException {

        String input = "";
        String password = "";
        String salt = "";

        AESGenerator g = new AESGenerator(password, salt);
        String encrypted = g.encrypt(input);
        System.out.println("encrypted: " + encrypted);
        System.out.println("decrypted: " + g.decrypt(encrypted));
    }
}
